# This file tests the filename manipulation routines.
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1995-1996 Sun Microsystems, Inc.
# Copyright (c) 1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: fileName.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

tcltest::testConstraint testsetplatform [string equal testsetplatform [info commands testsetplatform]]
tcltest::testConstraint testtranslatefilename [string equal testtranslatefilename [info commands testtranslatefilename]]

global env
if {[tcltest::testConstraint testsetplatform]} {
    set platform [testgetplatform]
}

test filename-1.1 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype /
} absolute
test filename-1.2 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype /foo
} absolute
test filename-1.3 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype foo
} relative
test filename-1.4 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype c:/foo
} relative
test filename-1.5 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype ~
} absolute
test filename-1.6 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype ~/foo
} absolute
test filename-1.7 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype ~foo
} absolute
test filename-1.8 {Tcl_GetPathType: unix} {testsetplatform} {
    testsetplatform unix
    file pathtype ./~foo
} relative

test filename-2.1 {Tcl_GetPathType: mac, denerate names} {testsetplatform} {
    testsetplatform mac
    file pathtype /
} relative
test filename-2.2 {Tcl_GetPathType: mac, denerate names} {testsetplatform} {
    testsetplatform mac
    file pathtype /.
} relative
test filename-2.3 {Tcl_GetPathType: mac, denerate names} {testsetplatform} {
    testsetplatform mac
    file pathtype /..
} relative
test filename-2.4 {Tcl_GetPathType: mac, denerate names} {testsetplatform} {
    testsetplatform mac
    file pathtype //.//
} relative
test filename-2.5 {Tcl_GetPathType: mac, denerate names} {testsetplatform} {
    testsetplatform mac
    file pathtype //.//../.
} relative
test filename-2.6 {Tcl_GetPathType: mac, tilde names} {testsetplatform} {
    testsetplatform mac
    file pathtype ~
} absolute
test filename-2.7 {Tcl_GetPathType: mac, tilde names} {testsetplatform} {
    testsetplatform mac
    file pathtype ~:
} absolute
test filename-2.8 {Tcl_GetPathType: mac, tilde names} {testsetplatform} {
    testsetplatform mac
    file pathtype ~:foo
} absolute
test filename-2.9 {Tcl_GetPathType: mac, tilde names} {testsetplatform} {
    testsetplatform mac
    file pathtype ~/
} absolute
test filename-2.10 {Tcl_GetPathType: mac, tilde names} {testsetplatform} {
    testsetplatform mac
    file pathtype ~/foo
} absolute
test filename-2.11 {Tcl_GetPathType: mac, unix-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype /foo
} absolute
test filename-2.12 {Tcl_GetPathType: mac, unix-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype /./foo
} absolute
test filename-2.13 {Tcl_GetPathType: mac, unix-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype /..//./foo
} absolute
test filename-2.14 {Tcl_GetPathType: mac, unix-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype /foo/bar
} absolute
test filename-2.15 {Tcl_GetPathType: mac, unix-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype foo/bar
} relative
test filename-2.16 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype :
} relative
test filename-2.17 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype :foo
} relative
test filename-2.18 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype foo:
} absolute
test filename-2.19 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype foo:bar
} absolute
test filename-2.20 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype :foo:bar
} relative
test filename-2.21 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype ::foo:bar
} relative
test filename-2.22 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype ~foo
} absolute
test filename-2.23 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype :~foo
} relative
test filename-2.24 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype ~foo:
} absolute
test filename-2.25 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype foo/bar:
} absolute
test filename-2.26 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype /foo:
} absolute
test filename-2.27 {Tcl_GetPathType: mac, mac-style names} {testsetplatform} {
    testsetplatform mac
    file pathtype foo
} relative

test filename-3.1 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype /
} volumerelative
test filename-3.2 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype \\
} volumerelative
test filename-3.3 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype /foo
} volumerelative
test filename-3.4 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype \\foo
} volumerelative
test filename-3.5 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype c:/
} absolute
test filename-3.6 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype c:\\
} absolute
test filename-3.7 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype c:/foo
} absolute
test filename-3.8 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype c:\\foo
} absolute
test filename-3.9 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype c:
} volumerelative
test filename-3.10 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype c:foo
} volumerelative
test filename-3.11 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype foo
} relative
test filename-3.12 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype //foo/bar
} absolute
test filename-3.13 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype ~foo
} absolute
test filename-3.14 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype ~
} absolute
test filename-3.15 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype ~/foo
} absolute
test filename-3.16 {Tcl_GetPathType: windows} {testsetplatform} {
    testsetplatform windows
    file pathtype ./~foo
} relative

test filename-4.1 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split /
} {/}
test filename-4.2 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split /foo
} {/ foo}
test filename-4.3 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split /foo/bar
} {/ foo bar}
test filename-4.4 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split /foo/bar/baz
} {/ foo bar baz}
test filename-4.5 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split foo/bar
} {foo bar}
test filename-4.6 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split ./foo/bar
} {. foo bar}
test filename-4.7 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split /foo/../././foo/bar
} {/ foo .. . . foo bar}
test filename-4.8 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split ../foo/bar
} {.. foo bar}
test filename-4.9 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split {}
} {}
test filename-4.10 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split .
} {.}
test filename-4.11 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split ../
} {..}
test filename-4.12 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split ../..
} {.. ..}
test filename-4.13 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split //foo
} {/ foo}
test filename-4.14 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split foo//bar
} {foo bar}
test filename-4.15 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split ~foo
} {~foo}
test filename-4.16 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split ~foo/~bar
} {~foo ./~bar}
test filename-4.17 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split ~foo/~bar/~baz
} {~foo ./~bar ./~baz}
test filename-4.18 {Tcl_SplitPath: unix} {testsetplatform} {
    testsetplatform unix
    file split foo/bar~/baz
} {foo bar~ baz}

if {[tcltest::testConstraint testsetplatform]} {
    testsetplatform $platform
}

test filename-4.19 {Tcl_SplitPath} {
    set oldDir [pwd]
    set res [catch {
	cd [temporaryDirectory]
	file mkdir tildetmp
	set nastydir [file join tildetmp ./~tilde]
	file mkdir $nastydir
	set norm [file normalize $nastydir]
	cd tildetmp
	cd ./~tilde
	glob -nocomplain *
	set idx [string first tildetmp $norm]
	set norm [string range $norm $idx end]
	# fix path away so all platforms are the same
	regsub {(.*):$} $norm {\1} norm
	regsub -all ":" $norm "/" norm
	# make sure we can delete the directory we created
	cd $oldDir
	file delete -force $nastydir
	set norm
    } err]
    cd $oldDir
    catch {file delete -force [file join [temporaryDirectory] tildetmp]}
    list $res $err
} {0 tildetmp/~tilde}

test filename-5.1 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:b
} {a: b}
test filename-5.2 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:b:c
} {a: b c}
test filename-5.3 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:b:c:
} {a: b c}
test filename-5.4 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:
} {a:}
test filename-5.5 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a::
} {a: ::}
test filename-5.6 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:::
} {a: :: ::}
test filename-5.7 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split :a
} {a}
test filename-5.8 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split :a::
} {a ::}
test filename-5.9 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split :
} {:}
test filename-5.10 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ::
} {::}
test filename-5.11 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split :::
} {:: ::}
test filename-5.12 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:::b
} {a: :: :: b}
test filename-5.13 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /a:b
} {/a: b}
test filename-5.14 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~:
} {~:}
test filename-5.15 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~/:
} {~/:}
test filename-5.16 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~:foo
} {~: foo}
test filename-5.17 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~/foo
} {~: foo}
test filename-5.18 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~foo:
} {~foo:}
test filename-5.19 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:~foo
} {a: :~foo}
test filename-5.20 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /
} {:/}
test filename-5.21 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a:b/c
} {a: :b/c}
test filename-5.22 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /foo
} {foo:}
test filename-5.23 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /a/b
} {a: b}
test filename-5.24 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /a/b/foo
} {a: b foo}
test filename-5.25 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a/b
} {a b}
test filename-5.26 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ./foo/bar
} {: foo bar}
test filename-5.27 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ../foo/bar
} {:: foo bar}
test filename-5.28 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split {}
} {}
test filename-5.29 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split .
} {:}
test filename-5.30 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ././
} {: :}
test filename-5.31 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ././.
} {: : :}
test filename-5.32 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ../
} {::}
test filename-5.33 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ..
} {::}
test filename-5.34 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ../..
} {:: ::}
test filename-5.35 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split //foo
} {foo:}
test filename-5.36 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split foo//bar
} {foo bar}
test filename-5.37 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~foo
} {~foo:}
test filename-5.38 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~
} {~:}
test filename-5.39 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split foo
} {foo}
test filename-5.40 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~/
} {~:}
test filename-5.41 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~foo/~bar
} {~foo: :~bar}
test filename-5.42 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split ~foo/~bar/~baz
} {~foo: :~bar :~baz}
test filename-5.43 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split foo/bar~/baz
} {foo bar~ baz}
test filename-5.44 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a/../b
} {a :: b}
test filename-5.45 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a/../../b
} {a :: :: b}
test filename-5.46 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split a/.././../b
} {a :: : :: b}
test filename-5.47 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /../bar
} {bar:}
test filename-5.48 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /./bar
} {bar:}
test filename-5.49 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split //.//.././bar
} {bar:}
test filename-5.50 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split /..
} {:/..}
test filename-5.51 {Tcl_SplitPath: mac} {testsetplatform} {
    testsetplatform mac
    file split //.//.././
} {://.//.././}

test filename-6.1 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /
} {/}
test filename-6.2 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /foo
} {/ foo}
test filename-6.3 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /foo/bar
} {/ foo bar}
test filename-6.4 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /foo/bar/baz
} {/ foo bar baz}
test filename-6.5 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split foo/bar
} {foo bar}
test filename-6.6 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split ./foo/bar
} {. foo bar}
test filename-6.7 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /foo/../././foo/bar
} {/ foo .. . . foo bar}
test filename-6.8 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split ../foo/bar
} {.. foo bar}
test filename-6.9 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split {}
} {}
test filename-6.10 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split .
} {.}
test filename-6.11 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split ../
} {..}
test filename-6.12 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split ../..
} {.. ..}
test filename-6.13 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split //foo
} {/ foo}
test filename-6.14 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split foo//bar
} {foo bar}
test filename-6.15 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /\\/foo//bar
} {//foo/bar}
test filename-6.16 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /\\/foo//bar
} {//foo/bar}
test filename-6.17 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split /\\/foo//bar
} {//foo/bar}
test filename-6.18 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split \\\\foo\\bar
} {//foo/bar}
test filename-6.19 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split \\\\foo\\bar/baz
} {//foo/bar baz}
test filename-6.20 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split c:/foo
} {c:/ foo}
test filename-6.21 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split c:foo
} {c: foo}
test filename-6.22 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split c:
} {c:}
test filename-6.23 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split c:\\
} {c:/}
test filename-6.24 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split c:/
} {c:/}
test filename-6.25 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split c:/./..
} {c:/ . ..}
test filename-6.26 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split ~foo
} {~foo}
test filename-6.27 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split ~foo/~bar
} {~foo ./~bar}
test filename-6.28 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split ~foo/~bar/~baz
} {~foo ./~bar ./~baz}
test filename-6.29 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split foo/bar~/baz
} {foo bar~ baz}
test filename-6.30 {Tcl_SplitPath: win} {testsetplatform} {
    testsetplatform win
    file split c:~foo
} {c: ./~foo}

test filename-7.1 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join / a
} {/a}
test filename-7.2 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join a b
} {a/b}
test filename-7.3 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join /a c /b d
} {/b/d}
test filename-7.4 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join /
} {/}
test filename-7.5 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join a
} {a}
test filename-7.6 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join {}
} {}
test filename-7.7 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join /a/ b
} {/a/b}
test filename-7.8 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join /a// b
} {/a/b}
test filename-7.9 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join /a/./../. b
} {/a/./.././b}
test filename-7.10 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join ~ a
} {~/a}
test filename-7.11 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join ~a ~b
} {~b}
test filename-7.12 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join ./~a b
} {./~a/b}
test filename-7.13 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join ./~a ~b
} {~b}
test filename-7.14 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join ./~a ./~b
} {./~a/~b}
test filename-7.15 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join a . b
} {a/./b}
test filename-7.16 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join a . ./~b
} {a/./~b}
test filename-7.17 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join //a b
} {/a/b}
test filename-7.18 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    file join /// a b
} {/a/b}

test filename-8.1 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a b
} {:a:b}
test filename-8.2 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join :a b
} {:a:b}
test filename-8.3 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a b:
} {b:}
test filename-8.4 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a: :b
} {a:b}
test filename-8.5 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a: :b:
} {a:b}
test filename-8.6 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a :: b
} {:a::b}
test filename-8.7 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a :: :: b
} {:a:::b}
test filename-8.8 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a ::: b
} {:a:::b}
test filename-8.9 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a: b:
} {b:}
test filename-8.10 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join /a/b
} {a:b}
test filename-8.11 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join /a/b c/d
} {a:b:c:d}
test filename-8.12 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join /a/b :c:d
} {a:b:c:d}
test filename-8.13 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join ~ foo
} {~:foo}
test filename-8.14 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join :: ::
} {:::}
test filename-8.15 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a: ::
} {a::}
test filename-8.16 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a {} b
} {:a:b}
test filename-8.17 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a::: b
} {a:::b}
test filename-8.18 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a : : :
} {:a}
test filename-8.19 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join :
} {:}
test filename-8.20 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join : a
} {:a}
test filename-8.21 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join a: :b/c
} {a:b/c}
test filename-8.22 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    file join :a :b/c
} {:a:b/c}

test filename-9.1 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join a b
} {a/b}
test filename-9.2 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join /a b
} {/a/b}
test filename-9.3 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join /a /b
} {/b}
test filename-9.4 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join c: foo
} {c:foo}
test filename-9.5 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join c:/ foo
} {c:/foo}
test filename-9.6 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join c:\\bar foo
} {c:/bar/foo}
test filename-9.7 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join /foo c:bar
} {c:bar}
test filename-9.8 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join ///host//share dir
} {//host/share/dir}
test filename-9.9 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join ~ foo
} {~/foo}
test filename-9.10 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join ~/~foo
} {~/~foo}
test filename-9.11 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join ~ ./~foo
} {~/~foo}
test filename-9.12 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join / ~foo
} {~foo}
test filename-9.13 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join ./a/ b c
} {./a/b/c}
test filename-9.14 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join ./~a/ b c
} {./~a/b/c}
test filename-9.15 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join // host share path
} {/host/share/path}
test filename-9.16 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join foo . bar
} {foo/./bar}
test filename-9.17 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join foo .. bar
} {foo/../bar}
test filename-9.18 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    file join foo/./bar
} {foo/./bar}
test filename-9.19 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    set res {}
    lappend res \
      [file join {C:\foo\bar}] \
      [file join C:/blah {C:\foo\bar}] \
      [file join C:/blah C:/blah {C:\foo\bar}]
} {C:/foo/bar C:/foo/bar C:/foo/bar}
test filename-9.19.1 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    set res {}
    lappend res \
      [file join {foo\bar}] \
      [file join C:/blah {foo\bar}] \
      [file join C:/blah C:/blah {foo\bar}]
} {foo/bar C:/blah/foo/bar C:/blah/foo/bar}
test filename-9.19.2 {Tcl_JoinPath: win} {testsetplatform winOnly} {
    testsetplatform win
    set res {}
    lappend res \
      [file join {foo\bar}] \
      [file join [pwd] {foo\bar}] \
      [file join [pwd] [pwd] {foo\bar}]
    string map [list [pwd] pwd] $res
} {foo/bar pwd/foo/bar pwd/foo/bar}
test filename-9.20 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    set res {}
    lappend res \
      [file join {/foo/bar}] \
      [file join /x {/foo/bar}] \
      [file join /x /x {/foo/bar}]
} {/foo/bar /foo/bar /foo/bar}
test filename-9.21 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    set res {}
    lappend res \
      [file join {/foo/bar}] \
      [file join drive: {/foo/bar}] \
      [file join drive: drive: {/foo/bar}]
} {foo:bar foo:bar foo:bar}
test filename-9.22 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    set res {}
    lappend res \
      [file join {foo:bar}] \
      [file join drive: {foo:bar}] \
      [file join drive: drive: {foo:bar}]
} {foo:bar foo:bar foo:bar}
test filename-9.23 {Tcl_JoinPath: win} {testsetplatform} {
    testsetplatform win
    set res {}
    lappend res \
      [file join {foo\bar}] \
      [file join C:/blah {foo\bar}] \
      [file join C:/blah C:/blah {foo\bar}]
    string map [list C:/blah ""] $res
} {foo/bar /foo/bar /foo/bar}
test filename-9.24 {Tcl_JoinPath: unix} {testsetplatform} {
    testsetplatform unix
    set res {}
    lappend res \
      [file join {foo/bar}] \
      [file join /x {foo/bar}] \
      [file join /x /x {foo/bar}]
    string map [list /x ""] $res
} {foo/bar /foo/bar /foo/bar}
test filename-9.25 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    set res {}
    lappend res \
      [file join {foo/bar}] \
      [file join drive: {foo/bar}] \
      [file join drive: drive: {foo/bar}]
    string map [list drive: ""] $res
} {:foo:bar foo:bar foo:bar}
test filename-9.26 {Tcl_JoinPath: mac} {testsetplatform} {
    testsetplatform mac
    set res {}
    lappend res \
      [file join {:foo:bar}] \
      [file join drive: {:foo:bar}] \
      [file join drive: drive: {:foo:bar}]
    string map [list drive: ""] $res
} {:foo:bar foo:bar foo:bar}

test filename-10.1 {Tcl_TranslateFileName} {testsetplatform} {
    testsetplatform unix
    list [catch {testtranslatefilename foo} msg] $msg
} {0 foo}
test filename-10.2 {Tcl_TranslateFileName} {testsetplatform} {
    testsetplatform windows
    list [catch {testtranslatefilename {c:/foo}} msg] $msg
} {0 {c:\foo}}
test filename-10.3 {Tcl_TranslateFileName} {testsetplatform} {
    testsetplatform windows
    list [catch {testtranslatefilename {c:/\\foo/}} msg] $msg
} {0 {c:\foo}}
test filename-10.4 {Tcl_TranslateFileName} {testsetplatform} {
    testsetplatform mac
    list [catch {testtranslatefilename foo} msg] $msg
} {0 :foo}
test filename-10.5 {Tcl_TranslateFileName} {testsetplatform} {
    testsetplatform mac
    list [catch {testtranslatefilename :~foo} msg] $msg
} {0 :~foo}
test filename-10.6 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "/home/test"
    testsetplatform unix
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 /home/test/foo}
test filename-10.7 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    unset env(HOME)
    testsetplatform unix
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {1 {couldn't find HOME environment variable to expand path}}
test filename-10.8 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "/home/test"
    testsetplatform unix
    set result [list [catch {testtranslatefilename ~} msg] $msg]
    set env(HOME) $temp
    set result
} {0 /home/test}
test filename-10.9 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "/home/test/"
    testsetplatform unix
    set result [list [catch {testtranslatefilename ~} msg] $msg]
    set env(HOME) $temp
    set result
} {0 /home/test}
test filename-10.10 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "/home/test/"
    testsetplatform unix
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 /home/test/foo}
test filename-10.11 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "Root:"
    testsetplatform mac
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 Root:foo}
test filename-10.12 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "Root:home"
    testsetplatform mac
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 Root:home:foo}
test filename-10.13 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "Root:home"
    testsetplatform mac
    set result [list [catch {testtranslatefilename ~::foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 Root:home::foo}
test filename-10.14 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "Root:home"
    testsetplatform mac
    set result [list [catch {testtranslatefilename ~} msg] $msg]
    set env(HOME) $temp
    set result
} {0 Root:home}
test filename-10.15 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "Root:home:"
    testsetplatform mac
    set result [list [catch {testtranslatefilename ~::foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 Root:home::foo}
test filename-10.16 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "Root:home::"
    testsetplatform mac
    set result [list [catch {testtranslatefilename ~::foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 Root:home:::foo}
test filename-10.17 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "\\home\\"
    testsetplatform windows
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 {\home\foo}}
test filename-10.18 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "\\home\\"
    testsetplatform windows
    set result [list [catch {testtranslatefilename ~/foo\\bar} msg] $msg]
    set env(HOME) $temp
    set result
} {0 {\home\foo\bar}}
test filename-10.19 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "c:"
    testsetplatform windows
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 c:foo}
test filename-10.20 {Tcl_TranslateFileName} {testtranslatefilename} {
    list [catch {testtranslatefilename ~blorp/foo} msg] $msg
} {1 {user "blorp" doesn't exist}}
test filename-10.21 {Tcl_TranslateFileName} {testsetplatform} {
    global env
    set temp $env(HOME)
    set env(HOME) "c:\\"
    testsetplatform windows
    set result [list [catch {testtranslatefilename ~/foo} msg] $msg]
    set env(HOME) $temp
    set result
} {0 {c:\foo}}
test filename-10.22 {Tcl_TranslateFileName} {testsetplatform} {
    testsetplatform windows
    list [catch {testtranslatefilename foo//bar} msg] $msg
} {0 {foo\bar}}

if {[tcltest::testConstraint testsetplatform]} {
    testsetplatform $platform
}

test filename-10.23 {Tcl_TranslateFileName} {unixOnly nonPortable} {
    # this test fails if ~ouster is not /home/ouster
    list [catch {testtranslatefilename ~ouster} msg] $msg
} {0 /home/ouster}
test filename-10.24 {Tcl_TranslateFileName} {unixOnly nonPortable} {
    # this test fails if ~ouster is not /home/ouster
    list [catch {testtranslatefilename ~ouster/foo} msg] $msg
} {0 /home/ouster/foo}


test filename-11.1 {Tcl_GlobCmd} {
    list [catch {glob} msg] $msg
} {1 {wrong # args: should be "glob ?switches? name ?name ...?"}}
test filename-11.2 {Tcl_GlobCmd} {
    list [catch {glob -gorp} msg] $msg
} {1 {bad option "-gorp": must be -directory, -join, -nocomplain, -path, -tails, -types, or --}}
test filename-11.3 {Tcl_GlobCmd} {
    list [catch {glob -nocomplai} msg] $msg
} {1 {wrong # args: should be "glob ?switches? name ?name ...?"}}
test filename-11.4 {Tcl_GlobCmd} {
    list [catch {glob -nocomplain} msg] $msg
} {1 {wrong # args: should be "glob ?switches? name ?name ...?"}}
test filename-11.5 {Tcl_GlobCmd} {
    list [catch {glob -nocomplain ~xyqrszzz} msg] $msg
} {0 {}}
test filename-11.6 {Tcl_GlobCmd} {
    list [catch {glob ~xyqrszzz} msg] $msg
} {1 {user "xyqrszzz" doesn't exist}}
test filename-11.7 {Tcl_GlobCmd} {
    list [catch {glob -- -nocomplain} msg] $msg
} {1 {no files matched glob pattern "-nocomplain"}}
test filename-11.8 {Tcl_GlobCmd} {
    list [catch {glob -nocomplain -- -nocomplain} msg] $msg
} {0 {}}
test filename-11.9 {Tcl_GlobCmd} {testsetplatform} {
    testsetplatform unix
    list [catch {glob ~\\xyqrszzz/bar} msg] $msg
} {1 {user "\xyqrszzz" doesn't exist}}
test filename-11.10 {Tcl_GlobCmd} {testsetplatform} {
    testsetplatform unix
    list [catch {glob -nocomplain ~\\xyqrszzz/bar} msg] $msg
} {0 {}}
test filename-11.11 {Tcl_GlobCmd} {testsetplatform} {
    testsetplatform unix
    list [catch {glob ~xyqrszzz\\/\\bar} msg] $msg
} {1 {user "xyqrszzz" doesn't exist}}
test filename-11.12 {Tcl_GlobCmd} {testsetplatform} {
    testsetplatform unix
    set home $env(HOME)
    unset env(HOME)
    set x [list [catch {glob ~/*} msg] $msg]
    set env(HOME) $home
    set x
} {1 {couldn't find HOME environment variable to expand path}}

if {[tcltest::testConstraint testsetplatform]} {
    testsetplatform $platform
}

test filename-11.13 {Tcl_GlobCmd} {
    list [catch {file join [lindex [glob ~] 0]} msg] $msg
} [list 0 [file join $env(HOME)]]

set oldpwd [pwd]
set oldhome $env(HOME)
cd [temporaryDirectory]
set env(HOME) [pwd]
file delete -force globTest
file mkdir globTest/a1/b1
file mkdir globTest/a1/b2
file mkdir globTest/a2/b3
file mkdir globTest/a3
close [open globTest/x1.c w]
close [open globTest/y1.c w]
close [open globTest/z1.c w]
close [open "globTest/weird name.c" w]
close [open globTest/a1/b1/x2.c w]
close [open globTest/a1/b2/y2.c w]

catch {close [open globTest/.1 w]}
catch {close [open globTest/x,z1.c w]}

test filename-11.14 {Tcl_GlobCmd} {
    list [catch {glob ~/globTest} msg] $msg
} [list 0 [list [file join $env(HOME) globTest]]]
test filename-11.15 {Tcl_GlobCmd} {
    list [catch {glob ~\\/globTest} msg] $msg
} [list 0 [list [file join $env(HOME) globTest]]]
test filename-11.16 {Tcl_GlobCmd} {
    list [catch {glob globTest} msg] $msg
} {0 globTest}

set globname "globTest"
set horribleglobname "glob\[\{Test"

test filename-11.17 {Tcl_GlobCmd} {unixOnly} {
    list [catch {lsort [glob -directory $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.17.1 {Tcl_GlobCmd} {pcOnly macOnly} {
    list [catch {lsort [glob -directory $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
        [file join $globname .1]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
if {[string equal $tcl_platform(platform) "windows"]} {
    if {[string index $tcl_platform(osVersion) 0] >= 5 \
      && ([lindex [file system [temporaryDirectory]] 1] == "NTFS")} {
	tcltest::testConstraint linkDirectory 1
    } else {
	tcltest::testConstraint linkDirectory 0
    }
} else {
    tcltest::testConstraint linkDirectory 1
}
if {[string equal $tcl_platform(platform) "windows"]} {
    tcltest::testConstraint symbolicLinkFile 0
} else {
    tcltest::testConstraint symbolicLinkFile 1
}
test filename-11.17.2 {Tcl_GlobCmd} {notRoot linkDirectory} {
    set dir [pwd]
    set ret "error in test"
    if {[catch {
	cd $globname
	file link -symbolic link a1
	cd $dir
	set ret [list [catch {
	    lsort [glob -directory $globname -join * b1]
	} msg] $msg]
    }]} {
	cd $dir
    }
    file delete [file join $globname link]
    set ret
} [list 0 [lsort [list [file join $globname a1 b1] \
  [file join $globname link b1]]]]
# Simpler version of the above test to illustrate a given bug.
test filename-11.17.3 {Tcl_GlobCmd} {notRoot linkDirectory} {
    set dir [pwd]
    set ret "error in test"
    if {[catch {
	cd $globname
	file link -symbolic link a1
	cd $dir
	set ret [list [catch {
	    lsort [glob -directory $globname -type d *]
	} msg] $msg]
    }]} {
	cd $dir
    }
    file delete [file join $globname link]
    set ret
} [list 0 [lsort [list [file join $globname a1] \
  [file join $globname a2] \
  [file join $globname a3] \
  [file join $globname link]]]]
# Make sure the bugfix isn't too simple.  We don't want
# to break 'glob -type l'.
test filename-11.17.4 {Tcl_GlobCmd} {notRoot linkDirectory} {
    set dir [pwd]
    set ret "error in test"
    if {[catch {
	cd $globname
	file link -symbolic link a1
	cd $dir
	set ret [list [catch {
	    lsort [glob -directory $globname -type l *]
	} msg] $msg]
    }]} {
	cd $dir
    }
    file delete [file join $globname link]
    set ret
} [list 0 [list [file join $globname link]]]
test filename-11.17.5 {Tcl_GlobCmd} {
    list [catch {lsort [glob -directory $globname -tails *.c]} msg] $msg
} [list 0 [lsort [list "weird name.c" x,z1.c x1.c y1.c z1.c]]]
test filename-11.17.6 {Tcl_GlobCmd} {
    list [catch {lsort [glob -directory $globname -tails *.c *.c]} msg] $msg
} [list 0 [lsort [concat [list "weird name.c" x,z1.c x1.c y1.c z1.c] \
  [list "weird name.c" x,z1.c x1.c y1.c z1.c]]]]
test filename-11.17.7 {Tcl_GlobCmd: broken link and glob -l} {linkDirectory} {
    set dir [pwd]
    set ret "error in test"
    if {[catch {
	cd $globname
	file mkdir nonexistent
	file link -symbolic link nonexistent
	file delete nonexistent
	cd $dir
	set ret [list [catch {
	    lsort [glob -nocomplain -directory $globname -type l *]
	} msg] $msg]
    }]} {
	cd $dir
    }
    file delete [file join $globname link]
    set ret
} [list 0 [list [file join $globname link]]]
test filename-11.17.8 {Tcl_GlobCmd: broken link and glob -l} {symbolicLinkFile} {
    set dir [pwd]
    set ret "error in test"
    if {[catch {
	cd $globname
	close [open "nonexistent" w]
	file link -symbolic link nonexistent
	file delete nonexistent
	cd $dir
	set ret [list [catch {
	    lsort [glob -nocomplain -directory $globname -type l *]
	} msg] $msg]
    }]} {
	cd $dir
    }
    file delete [file join $globname link]
    set ret
} [list 0 [list [file join $globname link]]]
test filename-11.18 {Tcl_GlobCmd} {unixOnly} {
    list [catch {lsort [glob -path $globname/ *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.18.1 {Tcl_GlobCmd} {pcOnly macOnly} {
    list [catch {lsort [glob -path $globname/ *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
        [file join $globname .1]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.19 {Tcl_GlobCmd} {unixOnly} {
    list [catch {lsort [glob -join -path \
	    [string range $globname 0 5] * *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.19.1 {Tcl_GlobCmd} {pcOnly macOnly} {
    list [catch {lsort [glob -join -path \
	    [string range $globname 0 5] * *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
        [file join $globname .1]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.20 {Tcl_GlobCmd} {
    list [catch {lsort [glob -type d -dir $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1]\
	[file join $globname a2]\
	[file join $globname a3]]]]
test filename-11.21 {Tcl_GlobCmd} {
    list [catch {lsort [glob -type d -path $globname *]} msg] $msg
} [list 0 [lsort [list $globname]]]

test filename-11.21.1 {Tcl_GlobCmd} {
    close [open {[tcl].testremains} w]
    set res [list [catch {lsort [glob -path {[tcl]} *]} msg] $msg]
    file delete -force {[tcl].testremains}
    set res
} [list 0 {{[tcl].testremains}}]

# Get rid of file/dir if it exists, since it will have
# been left behind by a previous failed run.
if {[file exists $horribleglobname]} {
    file delete -force $horribleglobname
}
file rename globTest $horribleglobname
set globname $horribleglobname

test filename-11.22 {Tcl_GlobCmd} {unixOnly} {
    list [catch {lsort [glob -dir $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.22.1 {Tcl_GlobCmd} {pcOnly macOnly} {
    list [catch {lsort [glob -dir $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
        [file join $globname .1]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.23 {Tcl_GlobCmd} {unixOnly} {
    list [catch {lsort [glob -path $globname/ *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.23.1 {Tcl_GlobCmd} {pcOnly macOnly} {
    list [catch {lsort [glob -path $globname/ *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
        [file join $globname .1]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.24 {Tcl_GlobCmd} {unixOnly} {
    list [catch {lsort [glob -join -path \
	    [string range $globname 0 5] * *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.24.1 {Tcl_GlobCmd} {pcOnly macOnly} {
    list [catch {lsort [glob -join -path \
	    [string range $globname 0 5] * *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1] [file join $globname a2]\
        [file join $globname .1]\
	[file join $globname a3]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-11.25 {Tcl_GlobCmd} {
    list [catch {lsort [glob -type d -dir $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1]\
	[file join $globname a2]\
	[file join $globname a3]]]]
test filename-11.25.1 {Tcl_GlobCmd} {
    list [catch {lsort [glob -type {d r} -dir $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1]\
  [file join $globname a2]\
  [file join $globname a3]]]]
test filename-11.25.2 {Tcl_GlobCmd} {
    list [catch {lsort [glob -type {d r w} -dir $globname *]} msg] $msg
} [list 0 [lsort [list [file join $globname a1]\
  [file join $globname a2]\
  [file join $globname a3]]]]
test filename-11.26 {Tcl_GlobCmd} {
    list [catch {glob -type d -path $globname *} msg] $msg
} [list 0 [list $globname]]
test filename-11.27 {Tcl_GlobCmd} {
    list [catch {glob -types abcde *} msg] $msg
} {1 {bad argument to "-types": abcde}}
test filename-11.28 {Tcl_GlobCmd} {
    list [catch {glob -types z *} msg] $msg
} {1 {bad argument to "-types": z}}
test filename-11.29 {Tcl_GlobCmd} {
    list [catch {glob -types {abcd efgh} *} msg] $msg
} {1 {only one MacOS type or creator argument to "-types" allowed}}
test filename-11.30 {Tcl_GlobCmd} {
    list [catch {glob -types {{macintosh type TEXT} \
	    {macintosh creator ALFA} efgh} *} msg] $msg
} {1 {only one MacOS type or creator argument to "-types" allowed}}
test filename-11.31 {Tcl_GlobCmd} {
    list [catch {glob -types} msg] $msg
} {1 {missing argument to "-types"}}
test filename-11.32 {Tcl_GlobCmd} {
    list [catch {glob -path hello -dir hello *} msg] $msg
} {1 {"-directory" cannot be used with "-path"}}
test filename-11.33 {Tcl_GlobCmd} {
    list [catch {glob -path} msg] $msg
} {1 {missing argument to "-path"}}
test filename-11.34 {Tcl_GlobCmd} {
    list [catch {glob -direct} msg] $msg
} {1 {missing argument to "-directory"}}
test filename-11.35 {Tcl_GlobCmd} {
    list [catch {glob -paths *} msg] $msg
} {1 {bad option "-paths": must be -directory, -join, -nocomplain, -path, -tails, -types, or --}}
# Test '-tails' flag to glob.
test filename-11.36 {Tcl_GlobCmd} {
    list [catch {glob -tails *} msg] $msg
} {1 {"-tails" must be used with either "-directory" or "-path"}}
test filename-11.37 {Tcl_GlobCmd} {
    list [catch {glob -type d -tails -path $globname *} msg] $msg
} [list 0 [list $globname]]
test filename-11.38 {Tcl_GlobCmd} {
    list [catch {glob -tails -path $globname *} msg] $msg
} [list 0 [list $globname]]
test filename-11.39 {Tcl_GlobCmd} {
    list [catch {glob -tails -join -path $globname *} msg] $msg
} [list 0 [list $globname]]
test filename-11.40 {Tcl_GlobCmd} {
    expr {[glob -dir [pwd] -tails *] == [glob *]}
} {1}
test filename-11.41 {Tcl_GlobCmd} {
    expr {[glob -dir [pwd] -tails *] != [glob -dir [pwd] *]}
} {1}
test filename-11.42 {Tcl_GlobCmd} {
    set res [list]
    foreach f [glob -dir [pwd] *] {
	lappend res [file tail $f]
    }
    expr {$res == [glob *]}
} {1}
test filename-11.43 {Tcl_GlobCmd} {
    list [catch {glob -t *} msg] $msg
} {1 {ambiguous option "-t": must be -directory, -join, -nocomplain, -path, -tails, -types, or --}}
test filename-11.44 {Tcl_GlobCmd} {
    list [catch {glob -tails -path hello -directory hello *} msg] $msg
} {1 {"-directory" cannot be used with "-path"}}
test filename-11.45 {Tcl_GlobCmd on root volume} {
    set res1 ""
    set res2 ""
    catch {
	set res1 [glob -dir [lindex [file volumes] 0] -tails *]
    }
    catch {
	set tmpd [pwd]
	cd [lindex [file volumes] 0]
	set res2 [glob *]
	cd $tmpd
    }
    expr {$res1 == $res2}
} {1}
test filename-11.46 {Tcl_GlobCmd} {
    list [catch {glob -types abcde -dir foo *} msg] $msg
} {1 {bad argument to "-types": abcde}}
test filename-11.47 {Tcl_GlobCmd} {
    list [catch {glob -types abcde -path foo *} msg] $msg
} {1 {bad argument to "-types": abcde}}
test filename-11.48 {Tcl_GlobCmd} {
    list [catch {glob -types abcde -dir foo -join * *} msg] $msg
} {1 {bad argument to "-types": abcde}}
test filename-11.49 {Tcl_GlobCmd} {
    list [catch {glob -types abcde -path foo -join * *} msg] $msg
} {1 {bad argument to "-types": abcde}}

file rename $horribleglobname globTest
set globname globTest
unset horribleglobname

test filename-12.1 {simple globbing} {unixOrPc} {
    list [catch {glob {}} msg] $msg
} {0 .}
test filename-12.1.1 {simple globbing} {unixOrPc} {
    list [catch {glob -types f {}} msg] $msg
} {1 {no files matched glob pattern ""}}
test filename-12.1.2 {simple globbing} {unixOrPc} {
    list [catch {glob -types d {}} msg] $msg
} {0 .}
test filename-12.1.3 {simple globbing} {unixOnly} {
    list [catch {glob -types hidden {}} msg] $msg
} {0 .}
test filename-12.1.4 {simple globbing} {pcOnly} {
    list [catch {glob -types hidden {}} msg] $msg
} {1 {no files matched glob pattern ""}}
test filename-12.1.5 {simple globbing} {pcOnly} {
    list [catch {glob -types hidden c:/} msg] $msg
} {1 {no files matched glob pattern "c:/"}}
test filename-12.1.6 {simple globbing} {pcOnly} {
    list [catch {glob c:/} msg] $msg
} {0 c:/}
test filename-12.2 {simple globbing} {macOnly} {
    list [catch {glob {}} msg] $msg
} {0 :}
test filename-12.2.1 {simple globbing} {macOnly} {
    list [catch {glob -types f {}} msg] $msg
} {1 {no files matched glob pattern ""}}
test filename-12.2.2 {simple globbing} {macOnly} {
    list [catch {glob -types d {}} msg] $msg
} {0 :}
test filename-12.2.3 {simple globbing} {macOnly} {
    list [catch {glob -types hidden {}} msg] $msg
} {1 {no files matched glob pattern ""}}
test filename-12.3 {simple globbing} {
    list [catch {glob -nocomplain \{a1,a2\}} msg] $msg
} {0 {}}

if {$tcl_platform(platform) == "macintosh"} {
    set globPreResult :globTest:
} else {
    set globPreResult globTest/
}
set x1 x1.c
set y1 y1.c
test filename-12.4 {simple globbing} {unixOrPc} {
    lsort [glob globTest/x1.c globTest/y1.c globTest/foo]
} "$globPreResult$x1 $globPreResult$y1"
test filename-12.5 {simple globbing} {
    list [catch {glob globTest\\/x1.c} msg] $msg
} "0 $globPreResult$x1"
test filename-12.6 {simple globbing} {
    list [catch {glob globTest\\/\\x1.c} msg] $msg
} "0 $globPreResult$x1"
test filename-12.7 {globbing at filesystem root} {unixOnly} {
    set res1 [glob -nocomplain /*]
    set res2 [glob -path / *]
    set equal [string equal $res1 $res2]
    if {!$equal} {
	lappend equal "not equal" $res1 $res2
    }
    set equal
} {1}
test filename-12.8 {globbing at filesystem root} {unixOnly} {
    set dir [lindex [glob -type d /*] 0]
    set first [string range $dir 0 1]
    set res1 [glob -nocomplain ${first}*]
    set res2 [glob -path $first *]
    set equal [string equal $res1 $res2]
    if {!$equal} {
	lappend equal "not equal" $res1 $res2
    }
    set equal
} {1}
test filename-12.9 {globbing at filesystem root} {winOnly} {
    # Can't grab just anything from 'file volumes' because we need a dir
    # that has subdirs - assume that C:/ exists across Windows machines.
    set dir [lindex [glob -type d C:/*] 0]
    set first [string range $dir 0 3]
    set res1 [glob -nocomplain ${first}*]
    set res2 [glob -path $first *]
    set equal [string equal $res1 $res2]
    if {!$equal} {
	lappend equal "not equal" $res1 $res2
    }
    set equal
} {1}

test filename-13.1 {globbing with brace substitution} {
    list [catch {glob globTest/\{\}} msg] $msg
} "0 $globPreResult"
test filename-13.2 {globbing with brace substitution} {
    list [catch {glob globTest/\{} msg] $msg
} {1 {unmatched open-brace in file name}}
test filename-13.3 {globbing with brace substitution} {
    list [catch {glob globTest/\{\\\}} msg] $msg
} {1 {unmatched open-brace in file name}}
test filename-13.4 {globbing with brace substitution} {
    list [catch {glob globTest/\{\\} msg] $msg
} {1 {unmatched open-brace in file name}}
test filename-13.5 {globbing with brace substitution} {
    list [catch {glob globTest/\}} msg] $msg
} {1 {unmatched close-brace in file name}}
test filename-13.6 {globbing with brace substitution} {
    list [catch {glob globTest/\{\}x1.c} msg] $msg
} "0 $globPreResult$x1"
test filename-13.7 {globbing with brace substitution} {
    list [catch {glob globTest/\{x\}1.c} msg] $msg
} "0 $globPreResult$x1"
test filename-13.8 {globbing with brace substitution} {
    list [catch {glob globTest/\{x\{\}\}1.c} msg] $msg
} "0 $globPreResult$x1"
test filename-13.9 {globbing with brace substitution} {
    list [lsort [catch {glob globTest/\{x,y\}1.c} msg]] $msg
} [list 0 [list $globPreResult$x1 $globPreResult$y1]]
test filename-13.10 {globbing with brace substitution} {
    list [lsort [catch {glob globTest/\{x,,y\}1.c} msg]] $msg
} [list 0 [list $globPreResult$x1 $globPreResult$y1]]
test filename-13.11 {globbing with brace substitution} {unixOrPc} {
    list [lsort [catch {glob globTest/\{x,x\\,z,z\}1.c} msg]] $msg
} {0 {globTest/x1.c globTest/x,z1.c globTest/z1.c}}
test filename-13.12 {globbing with brace substitution} {macOnly} {
    list [lsort [catch {glob globTest/\{x,x\\,z,z\}1.c} msg]] $msg
} {0 {:globTest:x1.c :globTest:x,z1.c :globTest:z1.c}}
test filename-13.13 {globbing with brace substitution} {
    lsort [glob globTest/{a,b,x,y}1.c]
} [list $globPreResult$x1 $globPreResult$y1]
test filename-13.14 {globbing with brace substitution} {unixOrPc} {
    lsort [glob {globTest/{x1,y2,weird name}.c}]
} {{globTest/weird name.c} globTest/x1.c}
test filename-13.15 {globbing with brace substitution} {macOnly} {
    lsort [glob {globTest/{x1,y2,weird name}.c}]
} {{:globTest:weird name.c} :globTest:x1.c}
test filename-13.16 {globbing with brace substitution} {unixOrPc} {
    lsort [glob globTest/{x1.c,a1/*}]
} {globTest/a1/b1 globTest/a1/b2 globTest/x1.c}
test filename-13.17 {globbing with brace substitution} {macOnly} {
    lsort [glob globTest/{x1.c,a1/*}]
} {:globTest:a1:b1 :globTest:a1:b2 :globTest:x1.c}
test filename-13.18 {globbing with brace substitution} {unixOrPc} {
    lsort [glob globTest/{x1.c,{a},a1/*}]
} {globTest/a1/b1 globTest/a1/b2 globTest/x1.c}
test filename-13.19 {globbing with brace substitution} {macOnly} {
    lsort [glob globTest/{x1.c,{a},a1/*}]
} {:globTest:a1:b1 :globTest:a1:b2 :globTest:x1.c}
test filename-13.20 {globbing with brace substitution} {unixOrPc} {
    lsort [glob globTest/{a,x}1/*/{x,y}*]
} {globTest/a1/b1/x2.c globTest/a1/b2/y2.c}
test filename-13.21 {globbing with brace substitution} {macOnly} {
    lsort [glob globTest/{a,x}1/*/{x,y}*]
} {:globTest:a1:b1:x2.c :globTest:a1:b2:y2.c}
test filename-13.22 {globbing with brace substitution} {
    list [catch {glob globTest/\{a,x\}1/*/\{} msg] $msg
} {1 {unmatched open-brace in file name}}

test filename-14.1 {asterisks, question marks, and brackets} {unixOrPc} {
    lsort [glob glo*/*.c]
} {{globTest/weird name.c} globTest/x,z1.c globTest/x1.c globTest/y1.c globTest/z1.c}
test filename-14.2 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob glo*/*.c]
} {{:globTest:weird name.c} :globTest:x,z1.c :globTest:x1.c :globTest:y1.c :globTest:z1.c}
test filename-14.3 {asterisks, question marks, and brackets} {unixOrPc} {
    lsort [glob globTest/?1.c]
} {globTest/x1.c globTest/y1.c globTest/z1.c}
test filename-14.4 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob globTest/?1.c]
} {:globTest:x1.c :globTest:y1.c :globTest:z1.c}

# The current directory could be anywhere; do this to stop spurious matches
file mkdir globTestContext
file rename globTest [file join globTestContext globTest]
set savepwd [pwd]
cd globTestContext

test filename-14.5 {asterisks, question marks, and brackets} {unixOrPc} {
    lsort [glob */*/*/*.c]
} {globTest/a1/b1/x2.c globTest/a1/b2/y2.c}
test filename-14.6 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob */*/*/*.c]
} {:globTest:a1:b1:x2.c :globTest:a1:b2:y2.c}

# Reset to where we were
cd $savepwd
file rename [file join globTestContext globTest] globTest
file delete globTestContext

test filename-14.7 {asterisks, question marks, and brackets} {unixOnly} {
    lsort [glob globTest/*]
} {globTest/a1 globTest/a2 globTest/a3 {globTest/weird name.c} globTest/x,z1.c globTest/x1.c globTest/y1.c globTest/z1.c}
test filename-14.7.1 {asterisks, question marks, and brackets} {pcOnly} {
    lsort [glob globTest/*]
} {globTest/.1 globTest/a1 globTest/a2 globTest/a3 {globTest/weird name.c} globTest/x,z1.c globTest/x1.c globTest/y1.c globTest/z1.c}
test filename-14.8 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob globTest/*]
} {:globTest:.1 :globTest:a1 :globTest:a2 :globTest:a3 {:globTest:weird name.c} :globTest:x,z1.c :globTest:x1.c :globTest:y1.c :globTest:z1.c}
test filename-14.9 {asterisks, question marks, and brackets} {unixOrPc} {
    lsort [glob globTest/.*]
} {globTest/. globTest/.. globTest/.1}
test filename-14.10 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob globTest/.*]
} {:globTest:.1}
test filename-14.11 {asterisks, question marks, and brackets} {unixOrPc} {
    lsort [glob globTest/*/*]
} {globTest/a1/b1 globTest/a1/b2 globTest/a2/b3}
test filename-14.12 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob globTest/*/*]
} {:globTest:a1:b1 :globTest:a1:b2 :globTest:a2:b3}
test filename-14.13 {asterisks, question marks, and brackets} {unixOrPc} {
    lsort [glob {globTest/[xyab]1.*}]
} {globTest/x1.c globTest/y1.c}
test filename-14.14 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob {globTest/[xyab]1.*}]
} {:globTest:x1.c :globTest:y1.c}
test filename-14.15 {asterisks, question marks, and brackets} {unixOrPc} {
    lsort [glob globTest/*/]
} {globTest/a1/ globTest/a2/ globTest/a3/}
test filename-14.16 {asterisks, question marks, and brackets} {macOnly} {
    lsort [glob globTest/*/]
} {:globTest:a1: :globTest:a2: :globTest:a3:}
test filename-14.17 {asterisks, question marks, and brackets} {
    global env
    set temp $env(HOME)
    set env(HOME) [file join $env(HOME) globTest]
    set result [list [catch {glob ~/z*} msg] $msg]
    set env(HOME) $temp
    set result
} [list 0 [list [file join $env(HOME) globTest z1.c]]]
test filename-14.18 {asterisks, question marks, and brackets} {unixOrPc} {
    list [catch {lsort [glob globTest/*.c goo/*]} msg] $msg
} {0 {{globTest/weird name.c} globTest/x,z1.c globTest/x1.c globTest/y1.c globTest/z1.c}}
test filename-14.19 {asterisks, question marks, and brackets} {macOnly} {
    list [catch {lsort [glob globTest/*.c goo/*]} msg] $msg
} {0 {{:globTest:weird name.c} :globTest:x,z1.c :globTest:x1.c :globTest:y1.c :globTest:z1.c}}
test filename-14.20 {asterisks, question marks, and brackets} {
    list [catch {glob -nocomplain goo/*} msg] $msg
} {0 {}}
test filename-14.21 {asterisks, question marks, and brackets} {
    list [catch {glob globTest/*/gorp} msg] $msg
} {1 {no files matched glob pattern "globTest/*/gorp"}}
test filename-14.22 {asterisks, question marks, and brackets} {
    list [catch {glob goo/* x*z foo?q} msg] $msg
} {1 {no files matched glob patterns "goo/* x*z foo?q"}}
test filename-14.23 {slash globbing} {unixOrPc} {
    glob /
} /
test filename-14.24 {slash globbing} {pcOnly} {
    glob {\\}
} /
test filename-14.25 {type specific globbing} {unixOnly} {
    list [catch {lsort [glob -dir globTest -types f *]} msg] $msg
} [list 0 [lsort [list \
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-14.25.1 {type specific globbing} {pcOnly macOnly} {
    list [catch {lsort [glob -dir globTest -types f *]} msg] $msg
} [list 0 [lsort [list \
        [file join $globname .1]\
	[file join $globname "weird name.c"]\
	[file join $globname x,z1.c]\
	[file join $globname x1.c]\
	[file join $globname y1.c] [file join $globname z1.c]]]]
test filename-14.26 {type specific globbing} {
    list [catch {glob -nocomplain -dir globTest -types {readonly} *} msg] $msg
} [list 0 {}]

unset globname

# The following tests are only valid for Unix systems.
# On some systems, like AFS, "000" protection doesn't prevent
# access by owner, so the following test is not portable.

catch {file attributes globTest/a1 -permissions 0000}
test filename-15.1 {unix specific globbing} {unixOnly nonPortable} {
    string tolower [list [catch {glob globTest/a1/*} msg]  $msg $errorCode]
} {1 {couldn't read directory "globtest/a1": permission denied} {posix eacces {permission denied}}}
test filename-15.2 {unix specific no complain: no errors} {unixOnly nonPortable} {
    glob -nocomplain globTest/a1/*
} {}
test filename-15.3 {unix specific no complain: no errors, good result} \
	{unixOnly nonPortable} {
    # test fails because if an error occur , the interp's result
    # is reset...
    glob -nocomplain globTest/a2 globTest/a1/* globTest/a3
} {globTest/a2 globTest/a3}

catch {file attributes globTest/a1 -permissions 0755}
test filename-15.4 {unix specific no complain: no errors, good result} \
	{unixOnly nonPortable} {
    # test fails because if an error occurs, the interp's result
    # is reset... or you don't run at scriptics where the
    # outser and welch users exists
    glob -nocomplain ~ouster ~foo ~welch
} {/home/ouster /home/welch}
test filename-15.4.1 {no complain: no errors, good result} {
    # test used to fail because if an error occurs, the interp's result
    # is reset... 
    string equal [glob -nocomplain ~wontexist ~blah ~] \
      [glob -nocomplain ~ ~blah ~wontexist]
} {1}
test filename-15.5 {unix specific globbing} {unixOnly nonPortable} {
    glob ~ouster/.csh*
} "/home/ouster/.cshrc"
catch {close [open globTest/odd\\\[\]*?\{\}name w]}
test filename-15.6 {unix specific globbing} {unixOnly} {
    global env
    set temp $env(HOME)
    set env(HOME) $env(HOME)/globTest/odd\\\[\]*?\{\}name
    set result [list [catch {glob ~} msg] $msg]
    set env(HOME) $temp
    set result
} [list 0 [list [lindex [glob ~] 0]/globTest/odd\\\[\]*?\{\}name]]
catch {file delete -force globTest/odd\\\[\]*?\{\}name}

# The following tests are only valid for Windows systems.
set oldDir [pwd]
if {$::tcltest::testConstraints(pcOnly)} {
    cd c:/
    file delete -force globTest
    file mkdir globTest
    close [open globTest/x1.BAT w]
    close [open globTest/y1.Bat w]
    close [open globTest/z1.bat w]
}

test filename-16.1 {windows specific globbing} {pcOnly} {
    lsort [glob globTest/*.bat]
} {globTest/x1.BAT globTest/y1.Bat globTest/z1.bat}
test filename-16.2 {windows specific globbing} {pcOnly} {
    glob c:
} c:
test filename-16.3 {windows specific globbing} {pcOnly} {
    glob c:\\\\
} c:/
test filename-16.4 {windows specific globbing} {pcOnly} {
    glob c:/
} c:/
test filename-16.5 {windows specific globbing} {pcOnly} {
    glob c:*bTest
} c:globTest
test filename-16.6 {windows specific globbing} {pcOnly} {
    glob c:\\\\*bTest
} c:/globTest
test filename-16.7 {windows specific globbing} {pcOnly} {
    glob c:/*bTest
} c:/globTest
test filename-16.8 {windows specific globbing} {pcOnly} {
    lsort [glob c:globTest/*.bat]
} {c:globTest/x1.BAT c:globTest/y1.Bat c:globTest/z1.bat}
test filename-16.9 {windows specific globbing} {pcOnly} {
    lsort [glob c:/globTest/*.bat]
} {c:/globTest/x1.BAT c:/globTest/y1.Bat c:/globTest/z1.bat}
test filename-16.10 {windows specific globbing} {pcOnly} {
    lsort [glob c:globTest\\\\*.bat]
} {c:globTest/x1.BAT c:globTest/y1.Bat c:globTest/z1.bat}
test filename-16.11 {windows specific globbing} {pcOnly} {
    lsort [glob c:\\\\globTest\\\\*.bat]
} {c:/globTest/x1.BAT c:/globTest/y1.Bat c:/globTest/z1.bat}

# some tests require a shared C drive

if {[catch {cd //[info hostname]/c}]} {
    set ::tcltest::testConstraints(sharedCdrive) 0
} else {
    set ::tcltest::testConstraints(sharedCdrive) 1
}

test filename-16.12 {windows specific globbing} {pcOnly sharedCdrive} {
    cd //[info hostname]/c
    glob //[info hostname]/c/*Test
} //[info hostname]/c/globTest
test filename-16.13 {windows specific globbing} {pcOnly sharedCdrive} {
    cd //[info hostname]/c
    glob "\\\\\\\\[info hostname]\\\\c\\\\*Test"
} //[info hostname]/c/globTest
test filename-16.14 {windows specific globbing} {pcOnly} {
    cd [lindex [glob -types d -dir C:/ *] 0]
    expr {[lsearch -exact [glob {{.,*}*}] ".."] != -1}
} {1}
test filename-16.15 {windows specific globbing} {pcOnly} {
    cd [lindex [glob -types d -dir C:/ *] 0]
    glob ..
} {..}
test filename-16.16 {windows specific globbing} {pcOnly} {
    file tail [lindex [glob "[lindex [glob -types d -dir C:/ *] 0]/.."] 0]
} {..}

test filename-17.1 {windows specific special files} {testsetplatform} {
    testsetplatform win
    list [file pathtype com1] [file pathtype con] [file pathtype lpt3] \
      [file pathtype prn] [file pathtype nul] [file pathtype aux] \
      [file pathtype foo]
} {absolute absolute absolute absolute absolute absolute relative}

test filename-17.2 {windows specific glob with executable} {winOnly} {
    makeDirectory execglob
    makeFile contents execglob/abc.exe
    makeFile contents execglob/abc.notexecutable
    set res [glob -nocomplain -dir [temporaryDirectory]/execglob \
      -tails -types x *]
    removeFile execglob/abc.exe
    removeFile execglob/abc.notexecutable
    removeDirectory execglob
    set res
} {abc.exe}

# cleanup
catch {file delete -force C:/globTest}
cd [temporaryDirectory]
file delete -force globTest
cd $oldpwd
set env(HOME) $oldhome
if {[tcltest::testConstraint testsetplatform]} {
    testsetplatform $platform
    catch {unset platform}
}
catch {unset oldhome temp result globPreResult}
::tcltest::cleanupTests
return
