/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.external.api;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.eclipse.lsat.common.graph.directed.editable.Node;
import org.eclipse.lsat.common.scheduler.graph.Dependency;
import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.graph.TaskDependencyGraph;
import org.eclipse.lsat.common.scheduler.graph.util.ResampleTimeStepData;
import org.eclipse.lsat.common.scheduler.schedule.Schedule;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledDependency;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledTask;
import org.eclipse.lsat.common.scheduler.schedule.Sequence;
import org.eclipse.lsat.external.api.model.ActionBase;
import org.eclipse.lsat.external.api.model.LsatData;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventBase;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventEntry;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventExit;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventTransfer;
import org.eclipse.lsat.external.api.model.ProductPropertyBase;
import org.eclipse.lsat.external.api.model.ProductPropertyBoolean;
import org.eclipse.lsat.external.api.model.ProductPropertyInteger;
import org.eclipse.lsat.external.api.model.ProductPropertyString;
import org.eclipse.lsat.product.productdata.EntryEvent;
import org.eclipse.lsat.product.productdata.ExitEvent;
import org.eclipse.lsat.product.productdata.ProductInstance;
import org.eclipse.lsat.product.productdata.ProductProperty;
import org.eclipse.lsat.product.productdata.TransferEvent;
import org.eclipse.lsat.scheduler.CollectedScheduleData;

import activity.Action;
import activity.Activity;
import activity.ActivitySet;
import activity.Claim;
import activity.Move;
import activity.RaiseEvent;
import activity.Release;
import activity.RequireEvent;
import activity.ResourceAction;
import activity.SchedulingType;
import activity.SimpleAction;
import activity.SyncBar;
import dispatching.ActivityDispatching;
import dispatching.Dispatch;
import expressions.Declaration;
import lsat_graph.ActionTask;
import lsat_graph.ClaimTask;
import lsat_graph.ClaimedByScheduledTask;
import lsat_graph.DispatchGraph;
import lsat_graph.PeripheralResource;
import lsat_graph.ReleaseTask;
import lsat_graph.StochasticAnnotation;
import machine.Axis;
import machine.BidirectionalPath;
import machine.FullMeshPath;
import machine.IResource;
import machine.Machine;
import machine.Peripheral;
import machine.PeripheralType;
import machine.Resource;
import machine.ResourceItem;
import machine.UnidirectionalPath;
import product.ProductDefinition;
import product.ValueType;
import setting.PhysicalSettings;
import setting.Settings;
import timing.Array;
import timing.Scalar;

public class ConvertToOpenApiSchemas {
    private Map<Dispatch, String> dispatchToId;

    private Map<Task, String> taskToId;

    private Map<ScheduledTask<Task>, String> scheduledTaskToId;

    private int uniqueId = 0;

    private double executionDataSampleFrequency;

    public ConvertToOpenApiSchemas(double sampleFrequency) {
        executionDataSampleFrequency = sampleFrequency;
        dispatchToId = new HashMap<>();
        taskToId = new HashMap<>();
        scheduledTaskToId = new HashMap<>();
    }

    public LsatData convert(Machine machine, Settings settings, ActivitySet activitySet, ActivityDispatching dispatching,
            TaskDependencyGraph<Task> graph, Schedule<Task> schedule,
            HashMap<Task, CollectedScheduleData> collectedData, List<ProductInstance> productData)
    {
        // unpack all resources from resource pools
        var allResources = activitySet.getActivities().stream().flatMap(act -> act.getNodes().stream())
                .filter(node -> node instanceof ResourceAction).map(node -> ((ResourceAction)node).getResource())
                .filter(Objects::nonNull).collect(Collectors.toSet());

        var exportLsatData = new LsatData();
        convertProductDefinitions(machine.getProductDefinitions(), exportLsatData);
        convertPeripheralTypes(machine.getPeripheralTypes(), exportLsatData);
        convertResources(allResources, settings, exportLsatData);
        convertSettings(settings, exportLsatData);
        convertActivitySet(activitySet, exportLsatData);
        convertDispatching(dispatching, exportLsatData);
        convertSchedule(schedule, collectedData, exportLsatData);
        convertGraph(graph, exportLsatData);
        if (productData != null) {
            convertProductData(productData, exportLsatData);
        }
        return exportLsatData;
    }

    private void convertProductDefinitions(Collection<ProductDefinition> productDefinitions, LsatData exportLsatData) {
        for (var productDefinition: productDefinitions) {
            var exportProductDefinition = new org.eclipse.lsat.external.api.model.ProductDefinition();
            exportProductDefinition.setName(productDefinition.getName());
            for (var productProperty: productDefinition.getPropertyDefinitions()) {
                var exportProductProperty = new org.eclipse.lsat.external.api.model.ProductPropertyDefinition();
                exportProductProperty.setName(productProperty.getName());
                if (productProperty.getValueType() == ValueType.BOOLEAN) {
                    exportProductProperty.setType(org.eclipse.lsat.external.api.model.ProductType.BOOLEAN);
                } else if (productProperty.getValueType() == ValueType.INTEGER) {
                    exportProductProperty.setType(org.eclipse.lsat.external.api.model.ProductType.INTEGER);
                } else if (productProperty.getValueType() == ValueType.STRING) {
                    exportProductProperty.setType(org.eclipse.lsat.external.api.model.ProductType.STRING);
                }
                exportProductDefinition.addPropertiesItem(exportProductProperty);
            }
            exportLsatData.putProductDefinitionsItem(productDefinition.getName(), exportProductDefinition);
        }
    }

    private void convertPeripheralTypes(Collection<PeripheralType> peripheralTypes, LsatData exportLsatData) {
        List<PeripheralType> sortedPeripheralTypes = peripheralTypes.stream()
                .sorted(Comparator.comparing(PeripheralType::getName)).collect(Collectors.toList());

        for (var peripheralType: sortedPeripheralTypes) {
            var exportPeripheralType = new org.eclipse.lsat.external.api.model.PeripheralType();
            exportPeripheralType.setName(peripheralType.getName());

            for (var axis: peripheralType.getAxes()) {
                var exportAxis = new org.eclipse.lsat.external.api.model.Axis();
                exportAxis.setUnit(axis.getUnit());
                for (var setpoint: axis.getSetPoints()) {
                    exportAxis.addSetPointItem(setpoint.getName());
                }
                exportPeripheralType.putAxesItem(axis.getName(), exportAxis);
            }

            for (var setpoint: peripheralType.getSetPoints()) {
                var exportSetpoint = new org.eclipse.lsat.external.api.model.SetPoint();
                exportSetpoint.setUnit(setpoint.getUnit());
                for (var axis: setpoint.getAxes()) {
                    exportSetpoint.addAxesItem(axis.getName());
                }
                exportPeripheralType.putSetPointsItem(setpoint.getName(), exportSetpoint);
            }

            for (var action: peripheralType.getActions()) {
                var exportAction = new org.eclipse.lsat.external.api.model.ActionType();
                exportAction.setName(action.getName());
                exportPeripheralType.addActionsItem(exportAction);
            }

            exportLsatData.putPeripheralTypesItem(peripheralType.getName(), exportPeripheralType);
        }
    }

    private void convertResources(Collection<IResource> iresources, Settings settings, LsatData exportLsatData) {
        List<IResource> sortedResources = iresources.stream()
                .sorted(Comparator.comparing(ConvertToOpenApiSchemas::getResourceName)).collect(Collectors.toList());

        for (var iresource: sortedResources) {
            Resource resource = null;
            if (iresource instanceof ResourceItem r) {
                resource = r.getResource();
            } else if (iresource instanceof Resource r) {
                resource = r;
            }
            String name = getResourceName(iresource);

            var exportResource = new org.eclipse.lsat.external.api.model.Resource();
            exportResource.setName(name);

            // Add references to peripherals
            for (var peripheral: resource.getPeripherals()) {
                var physicalSettings = settings.getPhysicalSettings(iresource, peripheral);
                convert(iresource, peripheral, physicalSettings, exportLsatData);
                exportResource.addPeripheralIdsItem(getPeripheralFqn(iresource, peripheral));
            }

            // Set ResourceType
            switch (resource.getResourceType()) {
                case EVENT:
                    exportResource.setResourceType(org.eclipse.lsat.external.api.model.ResourceType.EVENT);
                    break;
                case PASSIVE:
                    exportResource.setResourceType(org.eclipse.lsat.external.api.model.ResourceType.PASSIVE);
                    break;
                case REGULAR:
                    exportResource.setResourceType(org.eclipse.lsat.external.api.model.ResourceType.REGULAR);
                    break;
            }

            exportLsatData.putResourcesItem(name, exportResource);
        }
    }

    private void convert(IResource iResource, Peripheral peripheral, PhysicalSettings physicalSettings,
            LsatData exportLsatData)
    {
        var exportPeripheral = new org.eclipse.lsat.external.api.model.Peripheral();
        exportPeripheral.setName(peripheral.getName());
        exportPeripheral.setResourceId(getResourceName(iResource));
        exportPeripheral.setPeripheralTypeId(peripheral.getType().getName());

        // Add profiles (names)
        for (var profile: peripheral.getProfiles()) {
            exportPeripheral.addProfilesItem(profile.getName());
        }

        // Add symbolic positions with physical locations
        var axes = peripheral.getType().getAxes();
        var sortedAxes = axes.stream().sorted(Comparator.comparing(Axis::getName)).collect(Collectors.toList());
        for (var symbolicPos: peripheral.getPositions()) {
            var positionPerAxis = new LinkedHashMap<String, Double>();
            for (var axis: sortedAxes) {
                var value = physicalSettings.getMotionSettings().get(axis).getLocationSettings()
                        .get(symbolicPos.getPosition(axis)).getDefault();
                positionPerAxis.put(axis.getName(), value.doubleValue());
            }
            exportPeripheral.putPositionsItem(symbolicPos.getName(), positionPerAxis);
        }

        // Add paths
        for (var path: peripheral.getPaths()) {
            var exportPath = new org.eclipse.lsat.external.api.model.Path();
            if (path instanceof UnidirectionalPath uPath) {
                exportPath.setPathType(org.eclipse.lsat.external.api.model.PathType.UNI_DIRECTIONAL);
                exportPath.addPointsItem(uPath.getSource().getName());
                exportPath.addPointsItem(uPath.getTarget().getName());
            } else if (path instanceof BidirectionalPath bPath) {
                exportPath.setPathType(org.eclipse.lsat.external.api.model.PathType.BI_DIRECTIONAL);
                for (var point: bPath.getEndPoints()) {
                    exportPath.addPointsItem(point.getName());
                }
            } else if (path instanceof FullMeshPath mPath) {
                exportPath.setPathType(org.eclipse.lsat.external.api.model.PathType.FULL_MESH);
                for (var point: mPath.getEndPoints()) {
                    exportPath.addPointsItem(point.getName());
                }
            }
            for (var profile: path.getProfiles()) {
                exportPath.addProfilesItem(profile.getName());
            }
            exportPeripheral.addPathsItem(exportPath);
        }

        // Add timing settings
        for (var timingEntry: physicalSettings.getTimingSettings()) {
            if (timingEntry.getValue() instanceof Scalar scalar) {
                var exportValue = new org.eclipse.lsat.external.api.model.ScalarValue();
                exportValue.setType("Scalar");
                exportValue.setValue(scalar.getValue().doubleValue());
                exportPeripheral.putTimingSettingsItem(timingEntry.getKey().getName(), exportValue);
            } else if (timingEntry.getValue() instanceof Array array) {
                var exportValues = new org.eclipse.lsat.external.api.model.ArrayValues();
                exportValues.setType("Array");
                for (var value: array.getValues()) {
                    exportValues.addValuesItem(value.doubleValue());
                }
                exportPeripheral.putTimingSettingsItem(timingEntry.getKey().getName(), exportValues);
            }
        }

        // Export Motion settings
        for (var settingEntry: physicalSettings.getMotionSettings()) {
            var axis = settingEntry.getKey();
            var motionSettings = settingEntry.getValue();
            var exportMotionSettings = new org.eclipse.lsat.external.api.model.MotionSetting();

            // Add location settings
            for (var locationEntry: motionSettings.getLocationSettings()) {
                var physicalLocation = locationEntry.getValue();
                var exportPhysicalLocation = new org.eclipse.lsat.external.api.model.PhysicalLocation();
                var value = physicalLocation.getDefault();
                var minValue = physicalLocation.getMin();
                var maxValue = physicalLocation.getMax();
                exportPhysicalLocation.setValue(value.doubleValue());
                if (minValue != null) {
                    exportPhysicalLocation.setMin(minValue.doubleValue());
                }
                if (maxValue != null) {
                    exportPhysicalLocation.setMax(maxValue.doubleValue());
                }
                exportMotionSettings.putLocationsItem(locationEntry.getKey().getName(), exportPhysicalLocation);
            }

            // Add motion profile settings
            for (var profileEntry: motionSettings.getProfileSettings()) {
                var profileSetting = profileEntry.getValue();
                var exportProfile = new org.eclipse.lsat.external.api.model.Profile();
                // exportProfile.setMotionProfile(profileSetting.getMotionProfile());
                for (var argEntry: profileSetting.getMotionArguments()) {
                    var argName = argEntry.getKey();
                    var argValue = argEntry.getValue().evaluate().doubleValue();
                    exportProfile.putMotionArgumentsItem(argName, argValue);
                }
                exportMotionSettings.putProfilesItem(profileEntry.getKey().getName(), exportProfile);
            }

            // Add distance settings
            for (var distanceEntry: motionSettings.getDistanceSettings()) {
                var value = distanceEntry.getValue().evaluate().doubleValue();
                exportMotionSettings.putDistancesItem(distanceEntry.getKey().getName(), value);
            }

            exportPeripheral.putMotionSettingsItem(axis.getName(), exportMotionSettings);
        }

        // Add distances (names)
        for (var distance: peripheral.getDistances()) {
            exportPeripheral.addDistancesItem(distance.getName());
        }

        exportLsatData.putPeripheralsItem(getPeripheralFqn(iResource, peripheral), exportPeripheral);
    }

    private void convertSettings(Settings setting, LsatData exportLsatData) {
        List<Declaration> sortedDeclarations = setting.getDeclarations().stream()
                .sorted(Comparator.comparing(Declaration::getName)).collect(Collectors.toList());

        for (var decl: sortedDeclarations) {
            var value = decl.getExpression().evaluate().doubleValue();
            exportLsatData.putDeclarationsItem(decl.getName(), value);
        }
    }

    private void convertActivitySet(ActivitySet activitySet, LsatData exportLsatData) {
        for (var activity: activitySet.getActivities()) {
            var exportActivity = new org.eclipse.lsat.external.api.model.Activity();
            exportActivity.setName(activity.getName());

            // Sort the syncbar-actions in an activity by name to get a consistent output
            var sortedActions = new ArrayList<Node>(activity.getNodes());
            sortedActions.sort(Comparator
                    // Partition: non-SyncBar first (false) then SyncBar (true)
                    .comparing((Node n) -> n instanceof SyncBar)
                    // Only used when both are SyncBar: order by name
                    .thenComparing(a -> (a instanceof SyncBar) ? a.getName() : null,
                            Comparator.nullsFirst(String::compareTo) // null-safe, case-sensitive
                    ));

            for (var action: sortedActions) {
                // Actions can be: Claim, Release, Move, SimpleAction, RaiseEvent, RequireEvent, SyncBar
                if (action instanceof Claim claim) {
                    var exportClaim = createExportAction(org.eclipse.lsat.external.api.model.Claim.class, action);
                    exportClaim.setName(claim.getName());
                    exportClaim.setPassive(claim.isPassive());
                    exportClaim.setResourceId(getResourceName(claim.getResource()));
                    exportActivity.putActionsItem(claim.getName(), exportClaim);
                } else if (action instanceof Release release) {
                    var exportRelease = createExportAction(org.eclipse.lsat.external.api.model.Release.class, action);
                    exportRelease.setName(release.getName());
                    exportRelease.setResourceId(getResourceName(release.getResource()));
                    exportActivity.putActionsItem(release.getName(), exportRelease);
                } else if (action instanceof Move move) {
                    var exportMove = createExportAction(org.eclipse.lsat.external.api.model.Move.class, action);
                    exportMove.setName(move.getName());
                    if (move.getSchedulingType() == SchedulingType.ASAP) {
                        exportMove.setSchedulingType(org.eclipse.lsat.external.api.model.SchedulingType.ASAP);
                    }
                    if (move.getSchedulingType() == SchedulingType.ALAP) {
                        exportMove.setSchedulingType(org.eclipse.lsat.external.api.model.SchedulingType.ALAP);
                    }
                    exportMove.setProfile(move.getProfile().getName());
                    var sourcePosition = move.getSourcePosition();
                    if (sourcePosition != null) {
                        for (var positionEntry: sourcePosition.getAxisPosition()) {
                            exportMove.putSourcePositionItem(positionEntry.getKey().getName(),
                                    positionEntry.getValue().getName());
                        }
                    }
                    var targetPosition = move.getTargetPosition();
                    if (targetPosition != null) {
                        for (var positionEntry: targetPosition.getAxisPosition()) {
                            exportMove.putTargetPositionItem(positionEntry.getKey().getName(),
                                    positionEntry.getValue().getName());
                        }
                    }
                    var distance = move.getDistance();
                    if (distance != null) {
                        exportMove.setDistance(distance.getName());
                    }
                    exportMove.setStopAtTarget(move.isStopAtTarget());
                    exportMove.setPositionMove(move.isPositionMove());
                    exportMove.setResourceId(getResourceName(move.getResource()));
                    exportMove.setPeripheralId(getPeripheralFqn(move.getResource(), move.getPeripheral()));

                    exportActivity.putActionsItem(move.getName(), exportMove);
                } else if (action instanceof SimpleAction simpleAction) {
                    var exportAction = createExportAction(org.eclipse.lsat.external.api.model.PeripheralAction.class, action);
                    exportAction.setName(simpleAction.getName());
                    if (simpleAction.getSchedulingType() == SchedulingType.ASAP) {
                        exportAction.setSchedulingType(org.eclipse.lsat.external.api.model.SchedulingType.ASAP);
                    }
                    if (simpleAction.getSchedulingType() == SchedulingType.ALAP) {
                        exportAction.setSchedulingType(org.eclipse.lsat.external.api.model.SchedulingType.ALAP);
                    }
                    exportAction.setActionName(simpleAction.getType().getName());
                    exportAction.setResourceId(getResourceName(simpleAction.getResource()));
                    exportAction.setPeripheralId(getPeripheralFqn(simpleAction.getResource(), simpleAction.getPeripheral()));
                    exportActivity.putActionsItem(simpleAction.getName(), exportAction);
                } else if (action instanceof RaiseEvent event) {
                    var exportEvent = createExportAction(org.eclipse.lsat.external.api.model.RaiseEvent.class, action);
                    exportEvent.setName(event.getName());
                    exportEvent.setResourceId(getResourceName(event.getResource()));
                    exportActivity.putActionsItem(event.getName(), exportEvent);
                } else if (action instanceof RequireEvent event) {
                    var exportEvent = createExportAction(org.eclipse.lsat.external.api.model.RequireEvent.class, action);
                    exportEvent.setName(event.getName());
                    exportEvent.setResourceId(getResourceName(event.getResource()));
                    exportActivity.putActionsItem(event.getName(), exportEvent);
                } else if (action instanceof SyncBar bar) {
                    var exportSyncBar = createExportAction(org.eclipse.lsat.external.api.model.SyncBar.class, action);
                    exportSyncBar.setName(bar.getName());
                    exportActivity.putActionsItem(bar.getName(), exportSyncBar);
                }
            }

            // Set predecessor and successor moves after all actions (incl. moves) have been processed
            for (var action: sortedActions) {
                if (action instanceof Move move) {
                    var exportMove = (org.eclipse.lsat.external.api.model.Move)exportActivity.getActions().get(move.getName());
                    if (move.getPredecessorMove() != null) {
                        exportMove.setPredecessorMoveId(move.getPredecessorMove().getName());
                    }
                    if (move.getSuccessorMove() != null) {
                        exportMove.setSuccessorMoveId(move.getSuccessorMove().getName());
                    }
                }

                for (var edge: action.getIncomingEdges()) {
                    var exportDependency = new org.eclipse.lsat.external.api.model.ActivityDependency();
                    exportDependency.setSourceId(edge.getSourceNode().getName());
                    exportDependency.setTargetId(action.getName());
                    exportActivity.addDependenciesItem(exportDependency);
                }
            }

            exportLsatData.putActivitiesItem(activity.getName(), exportActivity);
        }
    }

    private void convertDispatching(ActivityDispatching dispatching, LsatData exportLsatData) {
        for (var dispatchGroup: dispatching.getDispatchGroups()) {
            var exportDispatchGroup = new org.eclipse.lsat.external.api.model.DispatchGroup();
            exportDispatchGroup.setName(dispatchGroup.getName());
            exportDispatchGroup.setOffset(dispatchGroup.getOffset().doubleValue());

            for (var dispatch: dispatchGroup.getDispatches()) {
                final String dispatchId = "D" + String.valueOf(uniqueId++);
                dispatchToId.put(dispatch, dispatchId);

                var exportDispatch = new org.eclipse.lsat.external.api.model.Dispatch();
                exportDispatch.setDescription(dispatch.getDescription());
                exportDispatch.setActivityId(dispatch.getActivity().getName());

                for (var attr: dispatch.getUserAttributes()) {
                    exportDispatch.putAttributesItem(attr.getKey().getName(), attr.getValue());
                }

                exportLsatData.putDispatchesItem(dispatchId, exportDispatch);

                exportDispatchGroup.addDispatchIdsItem(dispatchId);
            }

            exportLsatData.putDispatchGroupsItem(dispatchGroup.getName(), exportDispatchGroup);
        }
    }

    private void convertGraph(TaskDependencyGraph<Task> graph, LsatData exportLsatData) {
        var exportGraph = new org.eclipse.lsat.external.api.model.Graph();
        convert(graph.getEdges(), exportGraph);
        exportLsatData.setGraph(exportGraph);
    }

    private void convertSchedule(Schedule<Task> schedule, HashMap<Task, CollectedScheduleData> collectedData,
            LsatData exportLsatData)
    {
        convert(schedule.getSequences(), collectedData, exportLsatData);

        var exportSchedule = new org.eclipse.lsat.external.api.model.Schedule();
        exportSchedule.setName(schedule.getName());

        convertSequences(schedule.getSequences(), exportSchedule);
        convert(schedule.getEdges(), exportSchedule);

        exportLsatData.setSchedule(exportSchedule);
    }

    private void convertProductData(List<ProductInstance> productData, LsatData exportLsatData) {
        for (var product: productData) {
            var id = product.getProductID();
            var exportProduct = new org.eclipse.lsat.external.api.model.Product();
            for (var prop: product.getProperties()) {
                var value = prop.getPropertyValue();
                if (value instanceof Boolean boolValue) {
                    var exportProductProperties = createExportProductProperties(ProductPropertyBoolean.class, prop);
                    exportProductProperties.setValue(boolValue);
                    exportProduct.addPropertiesItem(exportProductProperties);
                } else if (value instanceof Integer integerValue) {
                    var exportProductProperties = createExportProductProperties(ProductPropertyInteger.class, prop);
                    exportProductProperties.setValue(integerValue);
                    exportProduct.addPropertiesItem(exportProductProperties);
                } else {
                    var exportProductProperties = createExportProductProperties(ProductPropertyString.class, prop);
                    exportProductProperties.setValue(value.toString());
                    exportProduct.addPropertiesItem(exportProductProperties);
                }
            }
            for (var lifeCycleEvent: product.getLifeCycle()) {
                if (lifeCycleEvent instanceof EntryEvent entryEvent) {
                    var entryTask = entryEvent.getEntry().getTask();
                    var entrySlot = entryEvent.getEntry().getSlot();
                    var exportEvent = createExportProductLifeCycleEvent(ProductLifeCycleEventEntry.class);
                    exportEvent.setTaskId(scheduledTaskToId.get(entryTask.getScheduledTask()));
                    if (entrySlot != null && !entrySlot.isEmpty()) {
                        exportEvent.setSlot(entrySlot);
                    }
                    exportProduct.addLifeCycleItem(exportEvent);
                } else if (lifeCycleEvent instanceof ExitEvent exitEvent) {
                    var exitTask = exitEvent.getExit().getTask();
                    var exitSlot = exitEvent.getExit().getSlot();
                    var exportEvent = createExportProductLifeCycleEvent(ProductLifeCycleEventExit.class);
                    exportEvent.setTaskId(scheduledTaskToId.get(exitTask.getScheduledTask()));
                    if (exitSlot != null && !exitSlot.isEmpty()) {
                        exportEvent.setSlot(exitSlot);
                    }
                    exportProduct.addLifeCycleItem(exportEvent);
                } else if (lifeCycleEvent instanceof TransferEvent transferEvent) {
                    var outTask = transferEvent.getTransfer().getOut().getTask();
                    var outSlot = transferEvent.getTransfer().getOut().getSlot();
                    var inTask = transferEvent.getTransfer().getIn().getTask();
                    var inSlot = transferEvent.getTransfer().getIn().getSlot();
                    var exportEvent = createExportProductLifeCycleEvent(ProductLifeCycleEventTransfer.class);
                    exportEvent.setOutTaskId(scheduledTaskToId.get(outTask.getScheduledTask()));
                    if (outSlot != null && !outSlot.isEmpty()) {
                        exportEvent.setOutSlot(outSlot);
                    }
                    exportEvent.setInTaskId(scheduledTaskToId.get(inTask.getScheduledTask()));
                    if (inSlot != null && !inSlot.isEmpty()) {
                        exportEvent.setInSlot(inSlot);
                    }
                    exportProduct.addLifeCycleItem(exportEvent);
                }
            }
            exportLsatData.putProductsItem(id, exportProduct);
        }
    }

    private void convert(Collection<Sequence<Task>> sequences, HashMap<Task, CollectedScheduleData> collectedData,
            LsatData exportLsatData)
    {
        for (var seq: sequences) {
            for (var task: seq.getScheduledTasks()) {
                final String id = "T" + String.valueOf(uniqueId++);
                scheduledTaskToId.put(task, id);
                taskToId.put(task.getTask(), id);

                var exportTask = new org.eclipse.lsat.external.api.model.Task();
                exportTask.setName(task.getName());
                exportTask.setStartTime(task.getStartTime().doubleValue());
                exportTask.setDuration(task.getDuration().doubleValue());
                if (task.getTask() instanceof ActionTask paTask) {
                    var action = paTask.getAction();
                    var activity = (Activity)action.eContainer();
                    exportTask.setActionId(action.getName());
                    exportTask.setActivityId(activity.getName());
                }
                if (task.getTask().eContainer() instanceof DispatchGraph dg) {
                    exportTask.setDispatchId(dispatchToId.get(dg.getDispatch()));
                }
                exportTask.setType(getTaskType(task));
                var executionDataParameters = task.getTask().getExecutionDataParameters();
                var axes = task.getTask().getExecutionData().keySet().stream().sorted().collect(Collectors.toList());
                var resampledData = ResampleTimeStepData.resample(task.getTask().getExecutionData(),
                        executionDataSampleFrequency);
                for (int index = 0; index < executionDataParameters.size(); index++) {
                    var exportExecutionData = new LinkedHashMap<String, List<Double>>();
                    for (var axis: axes) {
                        var values = getValuesAtLastIndex(resampledData, axis, index);
                        if (values.size() > 0) {
                            exportExecutionData.put(axis, values);
                        }
                    }
                    if (exportExecutionData.size() > 0) {
                        exportTask.putExecutionDataItem(executionDataParameters.get(index), exportExecutionData);
                    }
                }

                for (var aspect: task.getAspects()) {
                    if (aspect instanceof StochasticAnnotation stochasticAnnotation) {
                        var exportStochastics = new org.eclipse.lsat.external.api.model.StochasticAnnotation();
                        var exportStartTime = new org.eclipse.lsat.external.api.model.StochasticProperties();
                        exportStartTime.setMean(stochasticAnnotation.getMean().doubleValue());
                        exportStartTime.setMin(stochasticAnnotation.getMin().doubleValue());
                        exportStartTime.setMax(stochasticAnnotation.getMax().doubleValue());
                        exportStartTime.setStandardDeviation(stochasticAnnotation.getStandardDeviation().doubleValue());
                        var skewness = stochasticAnnotation.getSkewness();
                        if (skewness != null) {
                            exportStartTime.setSkewness(skewness.doubleValue());
                        }
                        exportStochastics.setStartTime(exportStartTime);
                        var confidenceInterval = stochasticAnnotation.getConfidenceInterval();
                        if (confidenceInterval != null) {
                            var exportConfidenceInterval = new org.eclipse.lsat.external.api.model.Bounds();
                            exportConfidenceInterval.setLower(confidenceInterval.getLower().doubleValue());
                            exportConfidenceInterval.setUpper(confidenceInterval.getUpper().doubleValue());
                            exportStochastics.setConfidenceInterval(exportConfidenceInterval);
                        }
                        exportStochastics.setCriticality(stochasticAnnotation.getCriticality().doubleValue());

                        var data = collectedData.get(task.getTask());
                        if (data != null) {
                            var exportMeasurements = new org.eclipse.lsat.external.api.model.StochasticMeasurements();
                            exportMeasurements.setStartTime(Arrays.stream(data.getStartTime()).boxed().collect(Collectors.toList()));
                            exportMeasurements.setDuration(Arrays.stream(data.getDuration()).boxed().collect(Collectors.toList()));
                            exportMeasurements.setCritical(Arrays.stream(data.getCritical()).boxed().collect(Collectors.toList()));
                            exportStochastics.setMeasurements(exportMeasurements);
                        }
                        exportTask.putAttributesItem("stochastics", exportStochastics);
                    } else if (aspect.getName().equals("Critical")) {
                        var exportAnnotation = new org.eclipse.lsat.external.api.model.TaskAnnotationBoolean();
                        exportAnnotation.setValue(true);
                        exportTask.putAttributesItem("critical", exportAnnotation);
                    } else {
                        var exportAnnotation = new org.eclipse.lsat.external.api.model.TaskAnnotationString();
                        exportAnnotation.setValue(aspect.toString());
                        exportTask.putAttributesItem(aspect.getName(), exportAnnotation);
                    }
                }

                exportLsatData.putTasksItem(id, exportTask);
            }
        }
    }

    private void convertSequences(Collection<Sequence<Task>> sequences,
            org.eclipse.lsat.external.api.model.Schedule exportSchedule)
    {
        for (var seq: sequences) {
            var exportSequence = new org.eclipse.lsat.external.api.model.Sequence();
            exportSequence.setName(seq.getName());
            var resourceName = seq.getResource().getContainer().getName();
            exportSequence.setResourceId(resourceName);

            if (seq.getResource() instanceof PeripheralResource peripheral) {
                var peripheralName = resourceName + "." + peripheral.getPeripheral().getName();
                exportSequence.setPeripheralId(peripheralName);
            }

            for (var t: seq.getScheduledTasks()) {
                exportSequence.addTaskIdsItem(scheduledTaskToId.get(t));
            }

            exportSchedule.addSequencesItem(exportSequence);
        }
    }

    private void convert(Collection<Dependency> dependencies, org.eclipse.lsat.external.api.model.Graph exportGraph) {
        for (var dependency: dependencies) {
            var sourceNode = dependency.getSourceNode();
            var targetNode = dependency.getTargetNode();

            var exportDependency = new org.eclipse.lsat.external.api.model.Dependency();
            exportDependency.setSourceId(taskToId.get(sourceNode));
            exportDependency.setTargetId(taskToId.get(targetNode));
            exportGraph.addDependenciesItem(exportDependency);
        }
    }

    private void convert(Collection<ScheduledDependency> dependencies,
            org.eclipse.lsat.external.api.model.Schedule exportSchedule)
    {
        var exportDependencies = new ArrayList<org.eclipse.lsat.external.api.model.Dependency>(dependencies.size());
        for (var dependency: dependencies) {
            var sourceNode = dependency.getSourceNode();
            var targetNode = dependency.getTargetNode();

            var exportDependency = new org.eclipse.lsat.external.api.model.Dependency();
            exportDependency.setSourceId(scheduledTaskToId.get(sourceNode));
            exportDependency.setTargetId(scheduledTaskToId.get(targetNode));
            exportDependencies.add(exportDependency);
        }

        // sort the dependencies by source and target to get a consistent list in the exported data
        exportDependencies.sort(Comparator
            .comparing(org.eclipse.lsat.external.api.model.Dependency::getSourceId)
            .thenComparing(org.eclipse.lsat.external.api.model.Dependency::getTargetId));

        for (var dependency: exportDependencies) {
            exportSchedule.addDependenciesItem(dependency);
        }
    }

    private <T extends ActionBase> T createExportAction(Class<T> exportActionType, Node node) {
        try {
            T exportAction = exportActionType.getDeclaredConstructor().newInstance();
            exportAction.setActionType(exportActionType.getSimpleName());

            // Add trace points if the action-type supports trace points (i.e., not for SyncBar)
            if (node instanceof Action action) {
                var entry = action.getEntry(); // entry, or start of an action
                if (entry != null) {
                    exportAction.setTracePointStart(entry.getValue());
                }
                var exit = action.getExit(); // exit, or end of an action
                if (exit != null) {
                    exportAction.setTracePointEnd(exit.getValue());
                }
            }
            return exportAction;
        } catch (Exception e) {
            throw new RuntimeException("Failed to create exportAction", e);
        }
    }

    private <T extends ProductPropertyBase> T createExportProductProperties(Class<T> exportActionType, ProductProperty prop) {
        try {
            T exportProductProperties = exportActionType.getDeclaredConstructor().newInstance();
            exportProductProperties.setPropertyType(getLastWord(exportActionType.getSimpleName()));
            exportProductProperties.setName(prop.getPropertyName());
            exportProductProperties.setTaskId(scheduledTaskToId.get(prop.getAction().getScheduledTask()));
            exportProductProperties.setAt(prop.getAt().toString());
            return exportProductProperties;
        } catch (Exception e) {
            throw new RuntimeException("Failed to create exportProductProperties", e);
        }
    }

    private <T extends ProductLifeCycleEventBase> T createExportProductLifeCycleEvent(Class<T> exportEventType) {
        try {
            T exportProductLifeCycleEvent = exportEventType.getDeclaredConstructor().newInstance();
            exportProductLifeCycleEvent.setEventType(getLastWord(exportEventType.getSimpleName()));
            return exportProductLifeCycleEvent;
        } catch (Exception e) {
            throw new RuntimeException("Failed to create exportProductLifeCycleEvent", e);
        }
    }

    private org.eclipse.lsat.external.api.model.TaskType getTaskType(ScheduledTask<?> task) {
        if (isEvent(task)) {
            return org.eclipse.lsat.external.api.model.TaskType.EVENT;
        } else if (isClaimOrRelease(task)) {
            return org.eclipse.lsat.external.api.model.TaskType.CLAIM_OR_RELEASE;
        } else {
            return org.eclipse.lsat.external.api.model.TaskType.PERIPHERAL_TASK;
        }
    }

    private static String getPeripheralFqn(IResource iresource, Peripheral peripheral) {
        String fqn = peripheral.getName();
        if (iresource instanceof ResourceItem r) {
            fqn = r.getResource().getName() + "." + r.getName() + "." + fqn;
        } else if (iresource instanceof Resource r) {
            fqn = r.getName() + "." + fqn;
        }
        return fqn;
    }

    private static String getResourceName(IResource iresource) {
        String name = iresource.getName();
        if (iresource instanceof ResourceItem r) {
            Resource resource = r.getResource();
            if (resource != null) {
                name = resource.getName() + "." + name;
            }
        } else if (iresource instanceof Resource r) {
            name = r.getName();
        }
        return name;
    }

    private static List<Double> getValuesAtLastIndex(Map<String, List<List<BigDecimal>>> data, String key,
            int lastIndex)
    {
        return Optional.ofNullable(data.get(key)).stream().flatMap(List::stream)
                .filter(innerList -> innerList.size() > lastIndex) // ensure index is valid
                .map(innerList -> innerList.get(lastIndex).doubleValue()).collect(Collectors.toList());
    }

    private static boolean isEvent(ScheduledTask<?> task) {
        return !task.getTask().getAspects().stream().toList().isEmpty();
    }

    private static boolean isClaimOrRelease(ScheduledTask<?> task) {
        return task instanceof ClaimedByScheduledTask || task.getTask() instanceof ClaimTask
                || task.getTask() instanceof ReleaseTask;
    }

    private static String getLastWord(String input) {
        if (input == null || input.isEmpty()) {
            return "";
        }
        // Regex: match an uppercase letter followed by lowercase letters or digits until the end
        var regex = "[A-Z][a-z0-9]*$";
        return input.replaceAll(".*(" + regex + ")", "$1");
    }
}
