/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.activity.diagram.services;

import static org.eclipse.lsat.common.queries.QueryableIterable.from;

import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.lsat.common.util.CollectionUtil;
import org.eclipse.lsat.timing.calculator.MotionCalculatorExtension;
import org.eclipse.lsat.timing.util.MoveHelper;
import org.eclipse.lsat.timing.util.TimingCalculator;
import org.eclipse.sirius.business.api.dialect.DialectManager;
import org.eclipse.sirius.business.api.session.Session;
import org.eclipse.sirius.business.api.session.SessionManager;
import org.eclipse.sirius.diagram.DDiagram;
import org.eclipse.sirius.diagram.DDiagramElement;
import org.eclipse.sirius.diagram.DNodeContainer;
import org.eclipse.sirius.diagram.DSemanticDiagram;
import org.eclipse.sirius.viewpoint.DSemanticDecorator;
import org.eclipse.sirius.viewpoint.description.DAnnotation;

import activity.Action;
import activity.Activity;
import activity.Claim;
import activity.Dependency;
import activity.EventAction;
import activity.Move;
import activity.PeripheralAction;
import activity.RaiseEvent;
import activity.Release;
import activity.RequireEvent;
import activity.ResourceAction;
import activity.SchedulingType;
import activity.SimpleAction;
import activity.SyncBar;
import activity.impl.ActivityQueries;
import activity.util.ActivityUtil;
import machine.IResource;
import machine.Peripheral;
import machine.ResourceType;
import setting.SettingUtil;
import setting.Settings;

public class ActivityServices {

    private final Map<Settings, TimingCalculator> timingCalculators = new LinkedHashMap<>();
    private static final String TIME_LABEL_KEY = "showTimeLabels";
    private static final String ANNOTATION_SOURCE = "org.eclipse.lsat.activity.diagram.services.ActivityServices";

    /**
     * Checks if PeripheralAction is ALAP.
     *
     * @param action
     * @return true if peripheral action is ALAP.
     */
    public static boolean isALAP(PeripheralAction action) {
        return action.getSchedulingType().equals(SchedulingType.ALAP);
    }

    /**
     * Get actions which use a certain resource
     */
    public static Collection<Action> getResourceActions(Activity activity, IResource resource) {
        ArrayList<Action> result = new ArrayList<Action>();
        CollectionUtil.addAll(result,
                ActivityQueries.getActionsFor(resource, ResourceAction.class, activity.getNodes()));
        return result;
    }

    public Collection<IResource> getResources(DSemanticDiagram element) {
        return getResources((Activity)element.getTarget());
    }

    public Collection<EventAction> getEvents(DSemanticDiagram element) {
        return getEvents((Activity)element.getTarget());
    }

    /**
     * Get the in-use resources in the activity
     */
    public static Collection<EventAction> getEvents(Activity activity) {
        if (activity == null) {
            return Collections.emptySet();
        }
        return from(activity.getNodes()).objectsOfKind(EventAction.class).asSet();
    }

    /**
     * Get the in-use resources in the activity
     */
    public static Collection<IResource> getResources(Activity activity) {
        if (activity == null) {
            return Collections.emptySet();
        }
        return from(activity.getNodes()).objectsOfKind(ResourceAction.class).xcollectOne(a -> a.getResource())
                .select(r -> ResourceType.EVENT != r.getResource().getResourceType()).asSet();
    }

    /**
     * Get the in-use resources in the activity
     */
    public static Collection<IResource> getEventResources(Activity activity) {
        if (activity == null) {
            return Collections.emptySet();
        }
        return from(activity.getNodes()).objectsOfKind(ResourceAction.class).xcollectOne(a -> a.getResource())
                .select(r -> r.getResource().getResourceType() == ResourceType.EVENT).asSet();
    }

    public static Collection<IResource> getResourcePlaceholder(DSemanticDiagram element, IResource resource) {
        return hasPeripherals(element, resource) ? Collections.emptyList() : Collections.singleton(resource);
    }

    public static boolean hasPeripherals(DSemanticDiagram element, IResource resource) {
        Activity activity = getActivity(element);
        return !ActivityQueries.getActionsFor(resource, PeripheralAction.class, activity.getNodes()).isEmpty();
    }

    public static Collection<Peripheral> getPeripherals(DSemanticDiagram element, IResource resource) {
        return getPeripherals(getActivity(element), resource);
    }

    public boolean isPassive(EObject element) {
        if (!(element instanceof DSemanticDecorator decorator)) {
            return false;
        }

        Activity activity = getActivity(decorator);
        IResource resource = getResource(decorator);

        if (activity == null || resource == null) {
            return false;
        }

        return ActivityUtil.getClaims(activity, resource).stream().allMatch(Claim::isPassive);
    }

    public static Collection<PeripheralAction> getPeripheralActions(EObject containerView) {
        if (!(containerView instanceof DSemanticDecorator)) {
            return Collections.emptyList();
        }

        DSemanticDecorator decorator = (DSemanticDecorator)containerView;
        EObject target = decorator.getTarget();

        if (!(target instanceof Peripheral)) {
            return Collections.emptyList();
        }

        Peripheral peripheral = (Peripheral)target;
        IResource resource = getResource(decorator);
        Activity activity = getActivity(decorator);

        if (activity == null || resource == null) {
            return Collections.emptyList();
        }

        return ActivityQueries.getActionsFor(resource, PeripheralAction.class, activity.getNodes())
                .select(a -> a.getPeripheral() == peripheral).asList();
    }

    public static IResource getResource(DSemanticDecorator element) {
        return getTarget(IResource.class, element);
    }

    public static Activity getActivity(DSemanticDecorator element) {
        return getTarget(Activity.class, element);
    }

    private static <T extends EObject> T getTarget(Class<T> clazz, DSemanticDecorator element) {
        while (!(clazz.isInstance(element.getTarget()))) {
            EObject parent = element.eContainer();
            if (!(parent instanceof DSemanticDecorator)) {
                return null;
            }
            element = (DSemanticDecorator)parent;
        }
        return clazz.cast(element.getTarget());
    }

    public static Collection<PeripheralAction> getPeripheralActions(Activity activity, IResource resource) {
        return ActivityQueries.getActionsFor(resource, PeripheralAction.class, activity.getNodes()).asOrderedSet();
    }

    /**
     * Get the in-use peripherals in the activity which belong to a certain resource
     */
    public static Collection<Peripheral> getPeripherals(Activity activity, IResource resource) {
        return ActivityQueries.getActionsFor(resource, PeripheralAction.class, activity.getNodes())
                .xcollectOne(pa -> pa.getPeripheral()).asOrderedSet();
    }

    public Collection<Claim> getClaims(DSemanticDiagram element, IResource resource) {
        return ActivityUtil.getClaims((Activity)element.getTarget(), resource);
    }

    public static void addClaimRelease(Activity activity, IResource resource) {
        ActivityUtil.addClaim(activity, resource);
        ActivityUtil.addRelease(activity, resource);
    }

    public Collection<Release> getReleases(DSemanticDiagram element, IResource resource) {
        return ActivityUtil.getReleases((Activity)element.getTarget(), resource);
    }

    public String getDescription(EventAction eventAction) {
        StringBuffer description = new StringBuffer(eventAction.getName()).append(": ");
        if (eventAction instanceof RequireEvent) {
            description.append("Require ");
        }
        if (eventAction instanceof RaiseEvent) {
            description.append("Raise ");
        }
        description.append(eventAction.getResource().fqn());
        return description.toString();
    }

    public String getTooltip(EventAction eventAction) {
        return getDescription(eventAction);
    }

    public String getDescription(PeripheralAction action) {
        StringBuilder description = new StringBuilder(action.getName()).append(": ");

        if (action instanceof SimpleAction) {
            description.append(((SimpleAction)action).getType().getName());
        } else if (action instanceof Move) {
            description.append(MoveHelper.getDescription((Move)action, false));
        }

        if (isALAP(action)) {
            description.append(System.lineSeparator()).append("ALAP");
        }

        if (shouldShowActivityTime(action)) {
            description.append(System.lineSeparator()).append(getActivityTime(action));
        }

        return description.toString();
    }

    public String getActivityTime(PeripheralAction action) {
        try {
            var settings = SettingUtil.getSettings(action.getPeripheral().eResource());
            var timingCalculator = timingCalculators.get(settings);
            if (timingCalculator == null) {
                var motionCalculator = MotionCalculatorExtension.getSelectedMotionCalculator();
                timingCalculator = new TimingCalculator(settings, motionCalculator);
                timingCalculators.put(settings, timingCalculator);
            }
            var value = timingCalculator.calculateDuration(action).setScale(3, RoundingMode.HALF_UP);
            return "time: " + value.toString();
        } catch (IllegalStateException e) {
            return "time: unknown";
        } catch (Exception e) {
            e.printStackTrace();
            return "time: unknown";
        }
    }

    public String getTooltip(PeripheralAction action) {
        StringBuffer description = new StringBuffer(action.getName()).append(": ");
        if (action instanceof SimpleAction) {
            description.append(action.getResource().fqn()).append('.');
            description.append(action.getPeripheral().getName()).append('.');
            description.append(((SimpleAction)action).getType().getName());
        } else if (action instanceof Move) {
            description.append(MoveHelper.getDescription((Move)action, true));
        }
        if (isALAP(action)) {
            description.append(" ").append("ALAP");
        }

        return description.toString();
    }

    public String getTooltip(SyncBar syncbar) {
        return syncbar.getName();
    }

    public String getTooltip(Claim claim) {
        StringBuffer description = new StringBuffer(claim.getName()).append(": ");
        description.append("Claim ");
        description.append(claim.getResource().fqn());
        return description.toString();
    }

    public String getTooltip(Release release) {
        StringBuffer description = new StringBuffer(release.getName()).append(": ");
        description.append("Release ");
        description.append(release.getResource().fqn());
        return description.toString();
    }

    public String getBeginLabel(Dependency dependency) {
        if (!(dependency.getSourceNode() instanceof Action)) {
            return null;
        }
        Action action = (Action)dependency.getSourceNode();
        if (null == action.getExit()) {
            return null;
        }
        StringBuffer label = new StringBuffer(action.getExit().getValue());
        if (null != action.getOuterExit()) {
            label.append("\n[").append(action.getOuterExit().getValue()).append("]");
        }
        return label.toString();
    }

    public String getEndLabel(Dependency dependency) {
        if (!(dependency.getTargetNode() instanceof Action)) {
            return null;
        }
        Action action = (Action)dependency.getTargetNode();
        if (null == action.getEntry()) {
            return null;
        }
        StringBuffer label = new StringBuffer();
        if (null != action.getOuterEntry()) {
            label.append("[").append(action.getOuterEntry().getValue()).append("]\n");
        }
        label.append(action.getEntry().getValue());
        return label.toString();
    }

    private boolean shouldShowActivityTime(PeripheralAction action) {
        Session session = SessionManager.INSTANCE.getSession(action);
        if (session == null) {
            return false;
        }

        boolean result = DialectManager.INSTANCE.getAllRepresentations(session).stream()
                .filter(rep -> rep instanceof DDiagram)
                .map(rep -> (DDiagram)rep)
                .filter(diagram -> diagramContainsAction(diagram, action))
                .findFirst()
                .map(ActivityServices::getShowTimeLabels)
                .orElse(false);

        return result;
    }

    private boolean diagramContainsAction(DDiagramElement element, PeripheralAction action) {
        if (action.equals(element.getTarget())) {
            return true;
        }

        // Recurse if element is a container type
        if (element instanceof org.eclipse.sirius.diagram.DNodeContainer) {
            DNodeContainer container = (DNodeContainer)element;
            for (DDiagramElement child : container.getOwnedDiagramElements()) {
                if (diagramContainsAction(child, action)) {
                    return true;
                }
            }
        }

        return false;
    }

    private boolean diagramContainsAction(DDiagram diagram, PeripheralAction action) {
        for (DDiagramElement de : diagram.getOwnedDiagramElements()) {
            if (diagramContainsAction(de, action)) {
                return true;
            }
        }
        return false;
    }

    public static void setShowTimeLabels(DDiagram diagram, boolean show) {
        DAnnotation ann = getOrCreateAnnotation(diagram);
        ann.getDetails().put(TIME_LABEL_KEY, Boolean.toString(show));
    }

    private static DAnnotation getOrCreateAnnotation(DDiagram diagram) {
        // Check if the annotation already exists
        for (DAnnotation ann : diagram.getEAnnotations()) {
            if (ANNOTATION_SOURCE.equals(ann.getSource())) {
                return ann;
            }
        }

        // Create new annotation
        DAnnotation annotation = org.eclipse.sirius.viewpoint.description.DescriptionFactory.eINSTANCE.createDAnnotation();
        annotation.setSource(ANNOTATION_SOURCE);

        // Add the reference to the diagram
        diagram.getEAnnotations().add(annotation);

        return annotation;
    }

    public static boolean getShowTimeLabels(DDiagram diagram) {
        for (DAnnotation ann : diagram.getEAnnotations()) {
            if (ANNOTATION_SOURCE.equals(ann.getSource())) {
                String value = ann.getDetails().get(TIME_LABEL_KEY);
                if (value != null) {
                    return Boolean.parseBoolean(value);
                }
            }
        }
        return false;
    }
}
